const tls = require('tls')

const skipTrustedCheckSettings = {
  rejectUnauthorized: false,
  requestCert: false,
  checkServerIdentity: function () {}
}

/**
 * Connection with KSC Open API server
 * Connection based on TCP socket
 * This connection is maintained permanently
 */
class BaseConnection {
  /**
   * @typedef {Object} Config
   * @property {string} openAPIHost - ip or domain name of KSC Open API server
   * @property {number} openAPIPort - port for KSC Open API server
   * @property {string} openAPIVersion - version of KSC Open API server
   * @property {string} [virtualServerName] - virtual server connection specific parameter
   * @property {Function} handshakeFunc - handshake method (login or other) for KSC Open API server
   * @property {object} tlsConfig - TLS config which will be filled with default settings in case of parameter absence
   * @property {array} certificatesInfo - Information about certificates used by KSC
   */

  /**
   * Create new connection to KSC Open API
   * @param {Config} config - ip or domain name of KSC Open API server
   */
  constructor ({
    openAPIHost,
    openAPIPort,
    openAPIVersion,
    handshakeFunc,
    virtualServerName,
    tlsConfig = {},
    certificatesInfo
  }) {
    this.uid = null
    this.kscUserId = null
    this.customHeaders = {}
    this.options = {}
    this.config = {
      openAPIHost,
      openAPIPort,
      openAPIVersion,
      handshakeFunc,
      virtualServerName,
      certificatesInfo
    }

    const defaultTlsConnectionConfig = {
      host: this.config.openAPIHost,
      port: this.config.openAPIPort,
      rejectUnauthorized: !this.config.certificatesInfo,
      requestCert: true,
      ca: global.trustedCertificates || [],
      checkServerIdentity: (host, cert) => {
        const baseValidationError = tls.checkServerIdentity(host, cert)

        if (!baseValidationError) return

        let error = null
        switch (baseValidationError.code) {
          case 'ERR_TLS_CERT_ALTNAME_INVALID':
            break
          default:
            error = baseValidationError
            error.code = 'WRONG_TRUSTED_CERT'
            break
        }
        return error
      }
    }
    this.tlsConnectionConfig = Object.assign(
      defaultTlsConnectionConfig,
      tlsConfig,
      process.env.skipTrustedCertificatesCheck
        ? skipTrustedCheckSettings
        : {})

    this.isBusy = false
    this.queue = []
    this.eventHandlers = []
  }

  /**
   * @description analysing each network response from KSC server
   * @memberof BaseConnection
   */
  handleNetworkResponse () { }

  /**
   * Register callback which fire on closing TCP socket`s event
   * @param {string} eventName - TCP socket`s event
   * @param {function} callback
   */
  on (eventName, callback) {
    if (!Array.isArray(this.eventHandlers[ eventName ])) {
      this.eventHandlers[ eventName ] = []
    }
    this.eventHandlers[ eventName ].push(callback)
  }

  /**
   * Match TLS certificates with certificates from Hosted Discovery Service
   */
  checkConnectionCertificates (peerCertificate) {
    const certificatesInfo = this.config.certificatesInfo
    if (certificatesInfo && certificatesInfo.length > 0) {
      if (!peerCertificate) {
        this.close()
        throw new Error('No TLS certificate')
      }
      const connectionCertificateFingerprint = peerCertificate.fingerprint256
      const normalizedFingerprint = connectionCertificateFingerprint.split(':').join('')
      const matchedWithHdsCertificate = certificatesInfo.some(function (item) {
        return normalizedFingerprint === item.Thumbprint
      })
      if (!matchedWithHdsCertificate) {
        this.close()
        throw new Error('HDS certificate information mismatch')
      }
    }
  }

  /**
   * Check is tcp connection event (close, error, etc.) handled
   * @param {string} eventName - TCP socket`s event
   */
  isEventHandled (eventName) {
    const eventHandlers = this.eventHandlers[ eventName ]
    return Array.isArray(eventHandlers) && eventHandlers.length > 0
  }

  /**
   * Set KSC user id to connection
   * @param kscUserId
   */
  setKscUserId (kscUserId) {
    this.kscUserId = kscUserId
  }

  /**
   * Set WorkspaceId to connection
   * @param id
   */
  setWsId (id) {
    this.wsId = id
  }

  /**
   * Set is ignore requests queue
   * @param isIgnore
   */
  setIsIgnoreRequestsQueue (isIgnore) {
    this.isIgnoreRequestsQueue = isIgnore
  }

  /**
   * Set login time
   * @param time
   */
  setLoginTime (time) {
    this.loginTime = time
  }

  /**
   * Set hosted instance id to connection
   * @param id
   */
  setHostedInstanceId (id) {
    this.hostedInstanceId = id
  }

  /**
   * @description Open connection with KSC by handshake or specific request
   * @memberof BaseConnection
  */
  open () {
    throw new Error('Method "open" is not implemented')
  }

  /**
   * @description closes KSC connection by request or interruption physical connection
   * @memberof BaseConnection
   */
  close () {
    throw new Error('Method "open" is not implemented')
  }

  /**
   * @description checks that current connection is alive, authenticated and can pass business requests
   * @readonly
   * @memberof BaseConnection
   */
  get isAlive () {
    throw new Error('Getter "isAlive" is not implemented')
  }
}

module.exports = BaseConnection
