const shortid = require('shortid')
const PermanentConnection = require('./permanent-connection')

/**
 * Storage for all opened TCP connections with KSC Open API server
 */
class ConnectionPool {
  constructor () {
    let pool = {}

    this.createConnection = createConnection
    this.getConnection = getConnection
    this.closeConnection = closeConnection
    this.closeAllConnections = closeAllConnections

    /**
     * Create permanent TCP connection with Open API server
     * @param {Array.<*>} args - connection options
     * @return {Promise.<string>} - connection`s unique id
     */
    async function createConnection (...args) {
      const [ConnectionConstructor] = args.splice(-1)
      const connection = new ConnectionConstructor(...args)
      const uid = shortid.generate()

      connection.uid = uid
      connection.on('close', () => {
        this.closeConnection(uid)
      })
      try {
        await connection.open()
        pool[uid] = connection
        return uid
      } catch (err) {
        if (
          err?.headers &&
          'www-authenticate' in err.headers &&
          err.headers['www-authenticate'].includes('totp') &&
          connection instanceof PermanentConnection
        ) {
          err.connectionId = uid
          pool[uid] = connection
        }
        throw err
      }
    }

    /**
     * Get connection from the pool
     * @param {string} uid -  connection`s unique id
     * @return {Connection}
     */
    function getConnection (uid) {
      return pool[uid]
    }

    /**
     * Close TCP connection and remove it from Pool
     * @param {string} uid - connection`s unique id
     */
    function closeConnection (uid) {
      const connection = pool[uid]
      if (connection) {
        connection.close()
        delete pool[uid]
      }
    }

    /**
     * Close all opened TCP connections
     */
    function closeAllConnections () {
      Object
        .keys(pool)
        .forEach(this.closeConnection)
    }
  }
}

module.exports = new ConnectionPool()
