const TokenConnection = require('./token-connection')
const https = require('https')
const http = require('http')
const {
  ERROR_START_STATUS,
  ERROR_UNAUTHORIZED_STATUS,
  ERROR_RETRY_STATUS,
  ERROR_NOT_FOUND_STATUS
} = require('../constants/ksc-status-codes')

const HANDLED_ERROR_CODES = [ERROR_RETRY_STATUS, ERROR_UNAUTHORIZED_STATUS]

class OAuthTokenConnection extends TokenConnection {
  /** Create new connection to KSC Open API based on Authorization header
   * @param {object} oAuthToken - token object containing access_token, id_token
   * @param {string} openAPIHost - ksc / ksc-proxy host
   * @param {string} openAPIPort - ksc / ksc-proxy port
   * @param {string} openAPIVersion - ksc / ksc-proxy version, ex.: "v1.0"
   * @param {array} certificatesInfo - Information about certificates used by KSC
   * @param {object} tlsConfig - contain ciphers, minVersion, maxVersion, ecdhCurve etc
   * @returns {any}
   */
  constructor (...args) {
    super(...args)
    this.oAuthToken = args[0].oAuthToken
    this.options = {
      ...this.options,
      ...this.tlsConnectionConfig,
      agent: new (process.env.XDR_KSC_PROXY_USE_HTTP === 'true' ? http : https).Agent(
        {
          keepAlive: true
        }
      )
    }
  }

  handleNetworkResponse (response = {}, requestOptions = {}) {
    if ((response.statusCode >= ERROR_START_STATUS) && !HANDLED_ERROR_CODES.includes(response.statusCode)) {
      runtime.logger.log('Fire error event on token connection', {
        requestPath: requestOptions.path,
        responseStatusCode: response.statusCode,
        responseStatusMessage: response.statusMessage
      })

      if (response.statusCode !== ERROR_NOT_FOUND_STATUS) {
        this.callEventHandlers('error')
      }
    }
  }

  async open () {
    // eslint-disable-next-line dot-notation
    this.customHeaders['Authorization'] = `Bearer ${this.oAuthToken.access_token}`

    if (this.config.handshakeFunc) {
      const result = await this.config.handshakeFunc(this)
      if (result.PxgError) {
        runtime.logger.error('[XDR]: Error in handshake function when opening connection:', result.PxgError.message)
        const err = new Error(result.PxgError.message)
        err.code = result.PxgError.code
        throw err
      }
      const { session } = result
      this.sessionId = session.id
      this.customHeaders['X-KSC-Aux-Session'] = this.sessionId
    }
  }

  updateToken (newToken) {
    // eslint-disable-next-line dot-notation
    this.customHeaders['Authorization'] = `Bearer ${newToken.access_token}`
    this.oAuthToken = newToken
  }

  close () {
    this.options.agent.destroy()
    this.options.agent = null
  }
}

module.exports = OAuthTokenConnection
