const BaseConnection = require('./base-connection')

const tls = require('tls')
const { OK_STATUS } = require('../constants/ksc-status-codes')

/**
 * Connection with KSC Open API server
 * Connection based on TCP socket
 * This connection is maintained permanently
 */
class PermanentConnection extends BaseConnection {
  /**
   * Create new connection to KSC Open API
   * @param {string} openAPIHost - ip or domain name of KSC Open API server
   * @param {number} openAPIPort - port for KSC Open API server
   * @param {string} openAPIVersion - version of KSC Open API server
   * @param {string} [virtualServerName] - virtual server connection specific parameter
   * @param {Function} handshakeFunc - handshake method (login or other) for KSC Open API server
   * @param {object} tlsConfig - TLS config which will be filled with default settings in case of parameter absence
   * @param {array} certificatesInfo - Information about certificates used by KSC
   */
  constructor (...args) {
    super(...args)
    this.socket = tls.connect(this.tlsConnectionConfig)
    this.socket.on('error', err => {
      runtime.logger.error(err)
    })
    this.options.createConnection = () => this.socket
    this.customHeaders['Connection'] = 'keep-alive'
  }

  /**
    * Open permanent TCP connection with Open API server
    * @return {Promise}
    */
  async open () {
    const { response, body } = await this.config.handshakeFunc(this)
    if (response.statusCode !== OK_STATUS) {
      throw new Error(body)
    }
  }

  on (eventName, callback) {
    super.on(eventName, callback)
    this.socket.on(eventName, callback)
  }

  get isAlive () {
    return this.socket && !this.socket.destroyed
  }

  /**
   * Close connection
   */
  close () {
    try {
      this.socket.destroy('TCP socket closed from Node.js Open API client')
    } catch (error) {
      runtime.logger.error(error)
    }
  }
}

module.exports = PermanentConnection
