const https = require('https')
const BaseConnection = require('./base-connection')
const { Session } = require('../open-api-client/open-api-client')
const { getResult } = require('../helpers/serializer')
const {
  OK_STATUS,
  ERROR_START_STATUS,
  ERROR_UNAUTHORIZED_STATUS,
  ERROR_RETRY_STATUS,
  ERROR_NOT_FOUND_STATUS
} = require('../constants/ksc-status-codes')

const HANDLED_ERROR_CODES = [ERROR_RETRY_STATUS]

/**
 * Connection with KSC Open API server
 * Connection based on TCP socket
 * This connection is maintained permanently
 */
class TokenConnection extends BaseConnection {
  constructor (...args) {
    super(...args)
    this.options = {
      ...this.options,
      ...this.tlsConnectionConfig,
      agent: new https.Agent(
        {
          keepAlive: true
        }
      )
    }
  }

  async open () {
    const { response, body } = await this.config.handshakeFunc(this)
    if (response.statusCode !== OK_STATUS) {
      throw new Error(body)
    }
    const { result } = getResult({ body })
    if (result.PxgError) {
      const err = new Error(result.PxgError.message)
      err.code = result.PxgError.code
      throw err
    }
    this.sessionId = result
    this.customHeaders['X-KSC-Session'] = this.sessionId
  }

  handleNetworkResponse (response = {}, requestOptions = {}) {
    if ((response.statusCode >= ERROR_START_STATUS) && !HANDLED_ERROR_CODES.includes(response.statusCode)) {
      runtime.logger.log('Fire error event on token connection', {
        requestPath: requestOptions.path,
        responseStatusCode: response.statusCode,
        responseStatusMessage: response.statusMessage
      })
      if (response.statusCode !== ERROR_NOT_FOUND_STATUS) {
        this.callEventHandlers('error')
      }
    }

    const codeToEvent = {
      [ERROR_UNAUTHORIZED_STATUS]: 'close'
    }

    switch (codeToEvent[response.statusCode]) {
      case 'close':
        this.callEventHandlers('close')
        break
    }
  }

  callEventHandlers (eventName) {
    if (this.isEventHandled(eventName)) {
      this.eventHandlers[eventName].forEach(fn => fn(this))
    }
  }

  get isAlive () {
    return Session.Ping({}, this)
      .then(() => true)
      .catch(() => false)
  }

  /**
   * Close connection
   */
  close () {
    return Session.EndSession(null, this)
      .then(() => {
        runtime.logger.info('Token connection closed successfully')
      })
      .catch((err) => {
        runtime.logger.warn(`Token connection was closed with error ${err}`)
      })
      .finally(() => {
        this.options?.agent?.destroy()
        this.options.agent = null
      })
  }
}

module.exports = TokenConnection
