const connectionQueue = require('./connection-queue')
const { ERROR_START_STATUS, ERROR_RETRY_STATUS } = require('../constants/ksc-status-codes')
const { getProtocolForKsc } = require('../open-api-client/utils/get-protocol-for-ksc')

const {
  OPEN_API_FILES_MAX_RETRY_COUNT: maxRetryCount = 5,
  OPEN_API_UPLOAD_FILE_PART_REQUEST_TIMEOUT: uploadFilePartRequestTimeout = 10000
} = process.env

const maxBytesPerChunk = 1024 * 1024 - 1
const protocol = getProtocolForKsc()

function getAgent () {
  return new protocol.Agent({
    keepAlive: false
  })
}

function uploadFilePart ({ chunk, from, to, path, fullFileLength }, connection) {
  const httpMethod = 'PUT'
  const headers = {
    'Content-Length': `${chunk.length}`,
    'Content-Range': `bytes ${from}-${to - 1}/${fullFileLength}`,
    Expect: '100-continue',
    ...connection.customHeaders
  }

  const options = {
    host: connection.config.openAPIHost,
    port: connection.config.openAPIPort,
    method: httpMethod,
    path,
    headers,
    ...connection.options
  }

  if (process.env.openApiAuthType === 'token' || process.env.openApiAuthType === 'oAuthToken') {
    options.agent = getAgent()
  }

  runtime.logger.log(`File Upload: [${connection.clientUid}] sending ${from}-${to - 1} bytes of ${fullFileLength}`)

  return new Promise(function (resolve, reject) {
    let retryCount = 0
    function request () {
      const req = protocol.request(options, async (response) => {
        if (response.statusCode >= ERROR_START_STATUS) {
          if (response.statusCode === ERROR_RETRY_STATUS) {
            retryCount++
            if (retryCount >= Number(maxRetryCount)) {
              reject(new Error(response.statusMessage))
            } else {
              connectionQueue.addRequestToQueue({ request, connection })
            }
          } else {
            reject(new Error(response.statusMessage))
          }
        } else {
          connectionQueue.refreshRequestsQueue({ connection })
          resolve(response.statusCode)
        }
      })

      req.once('continue', () => {
        req.end(chunk, 'binary')
      })

      req.once('error', (e) => {
        runtime.logger.error(
          `File Upload: [${connection.clientUid}] Error while sending ${from}-${to - 1} bytes of ${fullFileLength}`
        )
        reject(new Error(e))
      })
    }
    connectionQueue.addRequestToQueue({ request, connection })
  }).catch(error => {
    connectionQueue.refreshRequestsQueue({ connection })
    runtime.logger.error(`problem with request: ${error.message}, ${error.code}`)
    runtime.logger.error(error)
    runtime.metrics.counter({ name: 'upload_file_errors' }).increment()
  }).finally(() => {
    options.agent?.destroy()
  })
}

/**
 * Downloads file by given path using given connection instance
 * @typedef { object } Connection - OpenAPI Connection
 * @property {Array} customHeaders
 * @property {Array} options
 * @property {Object} config
 *
 * @param { object } props - Method props
 * @param { string } props.path - File path to download
 * @param { string } [props.method] - Http request method
 * @param { number } [props.chunkSize] - Size of data chunk to read
 * @param { ({ response: ServerResponse, contentLength: number, chunk: Buffer }) => {} } props.onChunkRecieved - File
 * chunk handler
 * @param { number } [props.contentLength] - Required if server response doesn't return 'Content-Length' header
 * @param { Connection } props.connection - OpenAPI Connection instance
 * @return { Promise<void> }
 */
function download ({
  path,
  method = 'GET',
  chunkSize = 100 * 1024,
  onChunkRecieved,
  contentLength: predefinedLength = null,
  connection: {
    customHeaders,
    options,
    config: {
      openAPIHost = null,
      openAPIPort = null
    } = {}
  }
}) {
  if (!onChunkRecieved) {
    throw new Error('No chunk handler passed to \'file.download\'')
  }

  const requestOptions = {
    host: openAPIHost,
    port: openAPIPort,
    method,
    path,
    headers: customHeaders,
    ...options
  }

  return new Promise((resolve, reject) => {
    const request = protocol.request(requestOptions, function (response) {
      const contentLength = predefinedLength || Number(response.headers['content-length'])
      let finished = false
      async function onStreamReadable () {
        const chunk = response.read(chunkSize)
        if (chunk !== null) {
          await onChunkRecieved({ response, contentLength, chunk })
        }
        if (finished) {
          resolve()
        } else {
          response.once('readable', onStreamReadable)
        }
      }
      response.once('readable', onStreamReadable)
      response.once('end', () => { finished = true })
    })
    request.on('error', reject)
    request.end()
  })
}

async function upload ({url, fileString, remainingFileLength}, connection) {
  let sentDataLength = 0
  const fullFileLength = fileString.length
  while (remainingFileLength > 0) {
    const chunk = fileString.substring(sentDataLength, sentDataLength + maxBytesPerChunk)
    const chunkLength = chunk.length
    remainingFileLength = remainingFileLength - chunkLength

    await uploadFilePart({
      chunk,
      from: sentDataLength,
      to: sentDataLength + chunkLength,
      path: url,
      fullFileLength
    }, connection)
    sentDataLength = sentDataLength + chunkLength
  }
}

module.exports = {
  download,
  uploadFilePart,
  upload
}
