const _ = require('lodash')

const RESULT_FIELD_NAME = 'PxgRetVal'
const ERROR_FIELD_NAME = 'PxgError'

const serializeRecursive = data => {
  _.forEach(data, (item, key) => {
    let isValueOriginal = item && item.type != null && item.value != null
    if (_.isPlainObject(item) && !isValueOriginal) {
      data[key] = {
        type: 'params',
        value: item
      }
      serializeRecursive(data[key].value)
    }
    if (Array.isArray(item)) {
      serializeRecursive(item)
    }
  })
}

const serialize = data => {
  let clonedData = _.cloneDeep(data)
  serializeRecursive(clonedData)
  return clonedData
}

const deserializeRecursive = data => {
  if (data && typeof data === 'object') {
    if (Array.isArray(data)) {
      return data.map(deserializeRecursive)
    } else {
      if (data.value !== undefined && data.type === 'params') {
        data = deserializeRecursive(data.value)
      } else {
        Object.keys(data)
          .forEach(property => {
            data[property] = deserializeRecursive(data[property])
          })
      }
    }
  }
  return data
}

const deserialize = data => {
  return deserializeRecursive(data)
}

/**
 * Serialize data to KSC format
 * @param _data - deserialized data
 * @returns {*}
 */
const serializeKscRequest = _data => {
  let data = _.cloneDeep(_data)
  serializeRecursive(data)
  return data
}

/**
 * Deserialize data from KSC format
 * @param body - serilized data or http response body
 * @param field - get specific field value
 * @returns {{result}}
 */
const deserializeKscRequest = ({ body }, field = RESULT_FIELD_NAME) => {
  if (body[ERROR_FIELD_NAME] !== undefined) {
    return getBody({ body })
  }
  const result = getResult({ body }, field)
  return deserializeRecursive(result)
}

/**
 * Get specific data field from response body
 * @param body - http response body
 * @param field - get specific field value
 * @returns {*}
 */
const getResult = ({ body }, field = RESULT_FIELD_NAME) => {
  if (body[ERROR_FIELD_NAME] !== undefined) {
    return getBody({ body })
  }

  let result = body[field]
  return { result }
}

/**
 * Get raw http body
 * @param body - http response body
 * @returns {{result: *}}
 */
const getBody = ({ body }) => {
  return { result: body }
}

const fromBuffer = ({ body }) => {
  if (!body) {
    return null
  }

  return { result: body.toString() }
}

module.exports = {
  serialize,
  deserialize,
  serializeKscRequest,
  deserializeKscRequest,
  getResult,
  getBody,
  fromBuffer
}
