/**
 * Async OpenAPI actions processing implementation
 */
class AsyncActionsQueue {
  constructor ({
    queueLimit = 5,
    resultTimeout = 60000,
    progressCallback = () => {},
    openApiClient
  } = {}) {
    this.actionsQueue = []
    this.queueLimit = queueLimit
    this.resultTimeout = resultTimeout
    this.openApiClient = openApiClient
    this.progressCallback = progressCallback
  }

  /**
   * Wait until connection queue is ready for new requests. Called in request hook
   * @param {object} Object.connection - Current request connection instance
   * @return { Promise<void> }
   */
  async isReady ({ connection: { uid } }) {
    const activeActions = this.actionsQueue
      .filter(({ ready, connectionId }) => connectionId === uid && !ready)
      .map(({ promise }) => promise)
    try {
      await Promise.all(activeActions.slice(0, 1 - this.queueLimit))
    } catch (error) {
      runtime.logger.error(error)
    }
  }

  /**
   * Create new action instance in async actions queue. Called in request hook
   * @param {object} Object.connection - Current request connection instance
   * @param { String } Object.requestId - Unique request id, which can be handled in response hook
   * @return {object} - Created action
   */
  createAction ({
    connection: { uid: connectionId } = {},
    requestId
  }) {
    let actionResolve
    let actionReject
    const action = {
      requestId,
      connectionId,
      promise: new Promise(function (resolve, reject) {
        [actionResolve, actionReject] = [resolve, reject]
      })
    }
    action.resolve = actionResolve
    action.reject = actionReject

    action.promise.finally(() => {
      action.ready = true
      action.timeout = setTimeout(() => {
        this.dropAction(action)
      }, this.resultTimeout)
    })

    this.actionsQueue.push(action)
    return action
  }

  /**
   * Get async action result
   * @param { String } Object.actionGuid - Async action result guid
   * @param { Promise<Object> } - Async action result
   */
  getActionResult ({ actionGuid }) {
    const action = this.actionsQueue.find(({ actionGuid: guid }) => guid === actionGuid)
    if (!action) return null

    clearTimeout(action.timeout)
    action.promise.finally(() => {
      this.dropAction(action)
    })

    return action.promise
  }

  /**
   * Process async action request response. Called in response hook
   * @param {object} response - Async action request response
   */
  processResponse (response) {
    const {
      actionGuid,
      sourceRequest: {
        connection,
        requestId
      }
    } = response

    const action = this.actionsQueue.find(({ requestId: id }) => id === requestId)
    if (!action) {
      // Action should be described in /open-api-client/async-action-commands.js
      throw new Error('Action was not found.')
    }

    if (!actionGuid) {
      action.reject(new Error('Unknown action'))
      this.dropAction(action)
      return
    }

    action.actionGuid = actionGuid
    this.processActionResult({ action, actionGuid, connection })
  }

  /**
   * Recursively check if async action is finished
   * @param {object} Object.action - Async action instance
   * @param { String } Object.actionGuid - Async action result guid
   * @param {object} Object.connection - Current request connection instance
   * @param { Number } Object.timeout - Timeout to perform next 'CheckActionState' request
   */
  processActionResult ({
    action,
    actionGuid,
    connection,
    timeout = 0
  }) {
    setTimeout(async () => {
      try {
        const { result } = await this.openApiClient.AsyncActionStateChecker.CheckActionState({
          wstrActionGuid: actionGuid
        }, connection)
        this.progressCallback({
          connection,
          actionGuid,
          result
        })
        if (result.bFinalized === true) {
          return result.bSuccededFinalized === true
            ? action.resolve(result)
            : action.reject(result)
        }

        if (result.PxgError) {
          throw new Error(result.PxgError)
        }

        this.processActionResult({ ...arguments[0], timeout: result.lNextCheckDelay })
      } catch (error) {
        action.reject(error)
      }
    }, timeout)
  }

  /**
   * Drop async action
   * @param action - Async action instance
   */
  dropAction (action) {
    if (!this.actionsQueue.includes(action)) return
    this.actionsQueue.splice(this.actionsQueue.indexOf(action), 1)
  }
}

let queueInstance
module.exports = {
  init: function (options) {
    queueInstance = new AsyncActionsQueue({
      ...options,
      openApiClient: require('./open-api-client')
    })
  },
  getInstance: () => queueInstance
}
