// TODO Node.js >= 8.3 — переехать с Object.assign на spread
const baseOpenAPIClient = require('./base-open-api-client')

/**
 * Получение выборки элементов, основанной на паттерне "итератор"
 */
module.exports = class BaseIterator extends baseOpenAPIClient {
  constructor () {
    super()
    let fetchRequest = null
    this.getFetchInstance = getFetchInstance

    /**
     * Декорация fetch-функции методами, на основании которых эта функция будет работать
     * @param {object} params
     * @param {Function} params.resetIterator - метод для создания выборки (открытия итератора)
     * @param {Function} params.getRecordCount - метода для получения количества всех элементов выборки
     * @param {Function} params.getRecordRange - метод для получения элементов из поределенного диапазона
     * @param {Function} params.releaseIterator - метод для закрытия итератора
     * @param {String} params.iteratorIdPropertyName
     * @param {boolean} isSingleton - флаг, необходимо ли генерировать новый инстанс fetch-функции
     * @return {Function} - метод для получения элементов из требуемого диапазона
     */
    function getFetchInstance ({ iteratorIdPropertyName, resetIterator, getRecordCount, getRecordRange, releaseIterator }, isSingleton = true) {
      if (!fetchRequest || !isSingleton) {
        fetchRequest = fetchRecords.bind(this, {
          iteratorIdPropertyName,
          resetIterator,
          getRecordCount,
          getRecordRange,
          releaseIterator
        })
      }
      return fetchRequest
    }

    /**
     * Получение элементов из требуемого диапазона
     * @param {object} params
     * @param {object} params.iteratorParams - параметры, с которыми осуществляется выборка (открывается итератор)
     * @param {number} params.nStart - номер первого элемента выборки
     * @param {number} params.nEnd - номер последнего элемента выборки
     * @param {boolean} getAllRecords - флаг, нужно ли выбрать все элементы
     * @param {boolean} closeIteratorOnFinish - флаг, нужно ли закрыть итератор по окончанию всех действий
     * @return {Promise.<*>}
     */
    async function fetchRecords ({
      iteratorIdPropertyName,
      resetIterator,
      getRecordCount,
      getRecordRange,
      releaseIterator
    },
    { iteratorParams,
      nStart = 0,
      nEnd = 0,
      getAllRecords = true,
      closeIteratorOnFinish = true
    }, connection) {
      // Открытие итератора
      const { result: iteratorId } = await resetIterator(iteratorParams, connection)
      const iteratorContainer = {
        [iteratorIdPropertyName]: iteratorId
      }
      // Формирование границ диапазона для выборки всех элементов
      if (getAllRecords) {
        nStart = 0
        ;({ result: nEnd } = await getRecordCount(iteratorContainer, connection))
      }
      // Выбор элементов требуемого диапазона
      const { result: itemsRange } = await getRecordRange(Object.assign({
        nStart,
        nEnd
      }, iteratorContainer), connection)
      // Закрытие итератора
      if (!closeIteratorOnFinish) {
        await releaseIterator(iteratorContainer, connection)
      }
      return {
        result: Object.assign(iteratorContainer, itemsRange)
      }
    }
  }
}
