const axios = require('axios')
const _ = require('lodash')
const LRU = require('lru-cache')
const shortId = require('shortid')
const { processRequest, processResponse } = require('./hooks')
const serializer = require('../helpers/index').serializer
const prepareRequestHeaders = require('../helpers/request-headers')
const { addRequestToQueue, refreshRequestsQueue } = require('../helpers/connection-queue')
const maxRetryCount = process.env.OPEN_API_RETRY_COUNT || 5
const retryTimeout = process.env.OPEN_API_RETRY_TIMEOUT || 500
const {
  ERROR_START_STATUS,
  ERROR_RETRY_STATUS,
  ERROR_FINISH_STATUS,
  ERROR_UNAUTHORIZED_STATUS
} = require('../constants/ksc-status-codes')
const { getConnectionInfo, getRequestPath } = require('./utils/request-utils')
const processUnauthorizedError = require('./xdr/processUnauthorizedError')
const { getProtocolForKsc } = require('../open-api-client/utils/get-protocol-for-ksc')
const OpenApiClassLogger = require('./utils/open-api-class-logger')

class HttpError extends Error {
  constructor ({ message, code, headers }) {
    super(message)
    this.code = code
    this.headers = headers
  }
}

const LRU_CACHE_MAX_SIZE_DEFAULT = 30
const LRU_CACHE_TTL_DEFAULT = 1000 * 10

const protocol = getProtocolForKsc()

/**
 * Base methods for communication with Open API server.
 */
class BaseOpenAPIClient {
  constructor () {
    // do interface publishing to clients (browser, plugin, etc.)
    this.isPublicInterface = true

    this.resultField = 'PxgRetVal'

    // carry context for usage in promise callback
    this.getBody = serializer.getBody.bind(this)
    this.getResult = serializer.getResult.bind(this)
    this.serialize = serializer.serializeKscRequest.bind(this)
    this.deserialize = serializer.deserializeKscRequest.bind(this)
    this.fromBuffer = serializer.fromBuffer.bind(this)

    this.options = {}

    this.logger = new OpenApiClassLogger(this.loggingDisabledMethods)
    this.memoizeMethods.forEach((methodName) => {
      this[methodName] = this.defaultMemoize(this[methodName])
    })
    this.cache = new LRU({
      max: this.maxCacheSize,
      ttl: LRU_CACHE_TTL_DEFAULT,
      fetchMethod: (key, staleValue, { context }) => {
        this.logger.log(
          context.command,
          `Updating cache for request ${getRequestPath({ ...context })}.`, null, getConnectionInfo(context.connection)
        )
        return this.mainRequest({ ...context })
      }
    })
  }

  /**
   * The maximum number of items that remain in the cache
   * @returns {number}
   */
  get maxCacheSize () {
    return LRU_CACHE_MAX_SIZE_DEFAULT
  }

  /**
   * Methods with disabled logging
   * @returns {string[]}
   */
  get loggingDisabledMethods () {
    return []
  }

  /**
   * Methods for memoization
   * @returns {[]}
   */
  get memoizeMethods () {
    return []
  }

  /**
   * Get default memoize function
   * @param {Function} fnc - function for memoization
   * @returns {Function}
   */
  defaultMemoize (fnc) {
    return _.memoize(fnc, (params, connection) => {
      return `${connection.uid}_${JSON.stringify(params)}`
    })
  }

  /**
   * Send http request using FIFO queue
   * @param {object} params
   * @param {string} params.command - backend action name
   * @param {string} params.httpMethod - http request method
   * @param {object} params.data - request body data
   * @param {object} params.headers - custom request headers
   * @param {Connection} params.connection - current tcp socket connection
   * @param {object} params.raw
   * @param {string} params.url
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @param {boolean} cacheConfig.useCache - get request result from cache or cache request result
   * @param {number} cacheConfig.ttl - max time to live for items in cache before they are considered stale
   * @param {boolean} cacheConfig.forceRefresh - the cached item will be re-fetched, even if it is not stale
   * @return {Promise}
   */
  baseRequest (
    params = {},
    cacheConfig = {
      useCache: false
    }
  ) {
    if (cacheConfig.useCache) {
      this.logger.log(
        params.command,
        `Cached request for ${getRequestPath({ ...params })}`, cacheConfig, getConnectionInfo(params.connection)
      )
      return this.cachedRequest(params, cacheConfig)
    }
    return this.mainRequest(params)
  }

  /**
   * Send http request using FIFO queue
   * @param {object} params
   * @param {string} params.command - backend action name
   * @param {string} params.httpMethod - http request method
   * @param {object} params.data - request body data
   * @param {object} params.headers - custom request headers
   * @param {Connection} params.connection - current tcp socket connection
   * @param {object} params.raw
   * @param {string} params.url
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @param {boolean} cacheConfig.useCache - get request result from cache or cache request result
   * @param {number} cacheConfig.ttl - max time to live for items in cache before they are considered stale
   * @param {boolean} cacheConfig.forceRefresh - the cached item will be re-fetched, even if it is not stale
   * @return {Promise}
   */
  cachedRequest (params = {}, cacheConfig) {
    return this.cache.fetch(`${params.connection.uid}_${params.command}_${JSON.stringify(params.data)}`, {
      ...cacheConfig,
      fetchContext: params
    })
  }

  /**
   * Send http request using FIFO queue
   * @param {object} params
   * @param {string} params.command - backend action name
   * @param {string} params.httpMethod - http request method
   * @param {object} params.data - request body data
   * @param {object} params.headers - custom request headers
   * @param {Connection} params.connection - current tcp socket connection
   * @param {object} params.raw
   * @param {string} params.url
   * @return {Promise}
   */
  mainRequest (params = {}) {
    const {
      raw,
      data: rawData,
      command: rawCommand,
      url,
      httpMethod,
      connection,
      headers,
      path
    } = params
    const self = this
    const requestId = shortId()
    return new Promise(async function (resolve, reject) {
      const { error, command, data } = await processRequest({
        requestId,
        connection,
        data: rawData,
        command: rawCommand,
        options: self.options,
        logger: self.logger
      })

      if (error) {
        reject(error)
        return
      }

      function request () {
        self
          .request({
            raw,
            command,
            url,
            httpMethod,
            data,
            headers,
            path,
            connection
          }, resolve, reject)
          .finally(onResponse)

        function onResponse (response) {
          if (!connection.isIgnoreRequestsQueue) {
            refreshRequestsQueue({ connection })
          }
          return response
        }
      }

      try {
        if (!connection.isIgnoreRequestsQueue) {
          addRequestToQueue({ request, connection })
        } else {
          request()
        }
      } catch (e) {
        reject(e)
      }
    }).then(response => processResponse({
      response,
      sourceRequest: {
        connection,
        request: () => this.baseRequest(...arguments),
        command: rawCommand,
        data: rawData,
        requestId
      },
      options: self.options,
      logger: self.logger
    }))
  }

  /**
   * Send http request
   * @param {object} props
   * @param {string} props.httpMethod - http request method
   * @param {boolean} [props.raw] - return raw bufer response
   * @param {string} [props.url] - url to be called, this or command is required
   * @param {string} [props.command] - backend action name to be called, this or url is required
   * @param {object} props.data - request body data
   * @param {object} props.headers - custom request headers
   * @param {Connection} props.connection - current tcp socket connection
   * @param {function} resolve - resolve promise function
   * @param {function} reject - reject promise function
   * @return {Promise.<object>}
   */
  request (
    {
      raw = false,
      command,
      url,
      httpMethod = 'POST',
      data = {},
      headers = {},
      path = null,
      connection
    } = {},
    resolve = function () {},
    reject = function () {}
  ) {
    const self = this
    const connectionInfo = getConnectionInfo(connection)

    return new Promise(function (resolve, reject) {
      let retryCount = 0
      function sendRequest () {
        const body = JSON.stringify(data)
          .replace(/("type":"long","value":)"(-?\d+)"/g, '$1' + '$2')
          .replace(/"long-structure(-?\d+)"/g, '{"type":"long","value":$1}')
          .replace(/"long(-?\d+)"/g, '$1')

        const options = {
          host: connection.config.openAPIHost,
          port: connection.config.openAPIPort,
          method: httpMethod,
          path: path || getRequestPath({ command, connection, url }),
          headers: prepareRequestHeaders({ connection, headers }),
          ...connection.options
        }

        options.headers['Content-Type'] = 'application/json'
        options.headers['Content-Length'] = Buffer.byteLength(body)
        self.logger.log(
          command,
          `Request ${options.path} is sent.`, null, connectionInfo
        )
        const request = protocol.request(options, function (response) {
          connection.handleNetworkResponse(response, options)
          self.logger.log(
            command,
            `Request ${options.path} is finished with status ${response.statusCode}.`, null, connectionInfo
          )

          const responseBody = []

          response.on('data', function (chunk) {
            responseBody.push(chunk)
          })

          response.on('end', async function () {
            if (response.statusCode === ERROR_RETRY_STATUS) {
              retryCount++
              if (retryCount < maxRetryCount) {
                self.logger.warn(`Got ${ERROR_RETRY_STATUS} status code. Retry #${retryCount}`, null, connectionInfo)
                setTimeout(sendRequest, retryTimeout)
                return
              }
            }

            if (process.env.IS_XDR) {
              if (response.statusCode === ERROR_UNAUTHORIZED_STATUS) {
                if (!processUnauthorizedError({ connection, sendRequest, retryCount, reject })) {
                  return
                }
              }
            }

            let body = Buffer.concat(responseBody)

            if (raw) {
              self.logger.log(
                command,
                `Got raw response with ${body.length} bytes`, null, connectionInfo
              )
              return resolve({ response, body })
            }

            const stringifiedChunk = body
              .toString()
              .replace(/"type":"long","value":-?\d+/g, (match, number, string) => {
                const arr = match.split(':')
                const l = arr.length
                arr[l - 1] = `"${arr[l - 1]}"`
                return arr.join(':')
              })
            try {
              body = JSON.parse(stringifiedChunk)
            } catch (err) {
              body = stringifiedChunk
            }
            if (response.statusCode >= ERROR_START_STATUS && response.statusCode < ERROR_FINISH_STATUS) {
              const err = new HttpError({ message: response.statusMessage, code: response.statusCode, headers: response.headers })
              reject(err)
            }
            resolve({ response, body })
          })
        })

        let requestSocket
        let onSocketSecureConnect
        const cleanSocketListener = () => {
          if (requestSocket && onSocketSecureConnect) {
            requestSocket.off('secureConnect', onSocketSecureConnect)
            requestSocket = null
            onSocketSecureConnect = null
          }
        }

        request.on('socket', function (socket) {
          requestSocket = socket
          onSocketSecureConnect = function () {
            if (!socket.isSessionReused()) {
              try {
                connection.checkConnectionCertificates(socket.getPeerCertificate())
              } catch (e) {
                request.emit('error', e)
              }
            }
          }
          socket.on('secureConnect', onSocketSecureConnect)
        })

        request.on('error', (error) => {
          cleanSocketListener()
          reject(error)
        })

        request.write(body, () => self.logger.log(command, 'data flushed', null, connectionInfo))

        request.end(() => {
          self.logger.log(command, 'stream finished', null, connectionInfo)
          cleanSocketListener()
        })
      }
      sendRequest()
    }).then(resolve)
      .catch(error => {
        self.logger.warn(`problem with request: ${error.message}`, null, connectionInfo)
        self.logger.error(error, null, connectionInfo)
        runtime
          .metrics
          .counter({
            name: 'openapi_request_errors',
            label: {
              command,
              host: connection.config.openAPIHost,
              port: connection.config.openAPIPort
            }
          })
          .increment()
        reject(error)
      })
  }

  sendURLRequestWithQueue ({ url }, connection) {
    return new Promise((resolve, reject) => {
      function request () {
        const httpsAgent = new protocol.Agent(connection.tlsConnectionConfig)
        axios({
          method: 'get',
          baseURL: `${protocol}://${connection.config.openAPIHost}:${connection.config.openAPIPort}`,
          headers: connection.customHeaders,
          url,
          httpsAgent
        })
          .then(resolve)
          .catch(reject)
          .finally(() => {
            if (!connection.isIgnoreRequestsQueue) {
              refreshRequestsQueue({ connection })
            }
          })
      }

      if (!connection.isIgnoreRequestsQueue) {
        addRequestToQueue({ request, connection })
      } else {
        request()
      }
    })
  }
}

module.exports = BaseOpenAPIClient
