/**
 * Iterator descriptor
 * @typedef { string } IteratorDescriptor
 *
 * Iterator placeholder
 * @typedef { string } iteratorPlaceholder
 *
 * OpenApi Client
 * @typedef { any } openApiClient
 *
 * OpenApi Connection
 * @typedef Connection
 * @property { string } clientUid - Client uid
 */

const ITERATOR_METHODS = [
  'SrvView.ResetIterator',
  'SrvView.ReleaseIterator'
]

/**
 * Iterators Queue implementation
 * Used to store and operate iterator descriptors
 *
 * @todo Add Redis keys expire timeouts
 */
module.exports = class IteratorsQueue {
  /**
   * @constructor
   * @param {object} props
   * @param {(({ connection, clearQueue }: {
   *   connection: Connection,
   *   clearQueue: () => Promise<any>
   * }) => any)[] } [props.onLimitExceededCallbacks] - Callbacks to call on iterators limit exceeded
   * @param { any } props.tempDataStore - Instance of Temp Data Store
   * @param { openApiClient } props.openApiClient - OpenApi Client instance
   */
  constructor ({
    onLimitExceededCallbacks = [],
    tempDataStore,
    openApiClient
  }) {
    this.onLimitExceededCallbacks = onLimitExceededCallbacks
    this.tempDataStore = tempDataStore
    this.openApiClient = openApiClient
  }

  /**
   * Returns iterators queue for given user UID
   * @param { string } clientUid - Unique user identifier
   * @returns { Promise<(iteratorPlaceholder | IteratorDescriptor)[]>} - Iterators
   */
  async getIteratorsForUser (clientUid) {
    const dataStoreKey = IteratorsQueue.createDataStoreKey(clientUid)
    const userIteratorsQueue = await this.tempDataStore.getList(dataStoreKey) || []
    return userIteratorsQueue
  }

  /**
   * Appends iterator descriptor to user's queue
   * @param {object} props
   * @param { string } props.clientUid - Unique user identifier
   * @param { string } props.command - OpenApi-method name
   * @param { any } props.response - OpenApi response
   * @return { Promise<void> }
   */
  async pushIterator ({
    clientUid,
    command,
    response: {
      body: {
        wstrIteratorId = null,
        wstrIterator = null,
        strAccessor = null,
        PxgRetVal = null
      } = {}
    }
  }) {
    const dataStoreKey = IteratorsQueue.createDataStoreKey(clientUid)
    const iteratorDescriptor = IteratorsQueue.createIteratorDescriptor({
      iterator: wstrIteratorId || wstrIterator || strAccessor || PxgRetVal,
      command
    })
    await this.tempDataStore.pushToList(dataStoreKey, iteratorDescriptor)
  }

  /**
   * Removes given iterator from user's iterator queue
   * @param {object} props
   * @param { string } props.clientUid - Unique user identifier
   * @param { string } props.command - OpenApi-method name
   * @param { string } props.iterator - Iterator value
   */
  async removeIterator ({ clientUid, command, iterator }) {
    const dataStoreKey = IteratorsQueue.createDataStoreKey(clientUid)
    const iteratorDescriptor = IteratorsQueue.createIteratorDescriptor({ iterator, command })
    await this.tempDataStore.removeFromListByValue(dataStoreKey, iteratorDescriptor)
  }

  /**
   * Appends iterator placeholder to user's iterator queue
   * @param {object} props
   * @param { string } props.clientUid - Unique user identifier
   * @param { string } props.requestId - OpenApi request id
   */
  async pushIteratorPlaceholder ({ clientUid, requestId }) {
    const dataStoreKey = IteratorsQueue.createDataStoreKey(clientUid)
    const iteratorPlaceholder = IteratorsQueue.createIteratorPlaceholder(requestId)
    await this.tempDataStore.pushToList(dataStoreKey, iteratorPlaceholder)
  }

  /**
   * Removes given iterator placeholder from user's iterator queue
   * @param {object} props
   * @param { string } props.clientUid - Unique user identifier
   * @param { string } props.requestId - OpenApi request id
   */
  async removeIteratorPlaceholder ({ clientUid, requestId }) {
    const dataStoreKey = IteratorsQueue.createDataStoreKey(clientUid)
    const iteratorPlaceholder = IteratorsQueue.createIteratorPlaceholder(requestId)
    await this.tempDataStore.removeFromListByValue(dataStoreKey, iteratorPlaceholder)
  }

  /**
   * Calls iterators limit exceeded callbacks
   * @param {object} props
   * @param { Connection } props.connection - OpenApi connection instance
   */
  async onLimitExceeded ({ connection }) {
    const clearQueue = () => this.clearQueue({ connection })
    this.onLimitExceededCallbacks.forEach(f => f({ connection, clearQueue }))
  }

  /**
   * Clears iterators queue for given connection
   * @param {object} props
   * @param { Connection } props.connection - OpenApi connection instance
   */
  async clearQueue ({ connection }) {
    const { clientUid } = connection
    const iterators = await this.getIteratorsForUser(clientUid)
    return Promise.all(iterators.map(iterator => {
      const [type, value] = iterator.split(':')
      switch (type) {
        case 'placeholder':
          return this.removeIteratorPlaceholder({ clientUid, requestId: value })
        case 'iterator':
          return this.openApiClient.SrvView.ReleaseIterator({ wstrIteratorId: value }, connection)
        case 'accessor':
          return this.openApiClient.ChunkAccessor.Release({ strAccessor: value }, connection)
      }
    }))
  }

  /**
   * Creates iterator placeholder from given request id
   * @param { string } requestId - OpenApi request id
   * @return { string } - Iterator placeholder
   */
  static createIteratorPlaceholder (requestId) {
    return `placeholder:${requestId}`
  }

  /**
   * Returns iterator type by given method name
   * @param { string } methodName - OpenApi method name
   * @return { 'iterator' | 'accessor' } - Iterator type
   */
  static getIteratorTypeByCommand (methodName) {
    return ITERATOR_METHODS.includes(methodName)
      ? 'iterator'
      : 'accessor'
  }

  /**
   * Returns data-store key for given user UID
   * @param { string } uid - Unique user identifier
   * @return { string } Data-store key
   */
  static createDataStoreKey (uid) {
    return `iterators-queue:${uid}`
  }

  /**
   * Creates iterator descriptor including data
   * required to close itreator later on
   * @param {object} props
   * @param { string } props.iterator - Iterator
   * @param { string } props.command - Method name
   */
  static createIteratorDescriptor ({ iterator, command }) {
    return `${IteratorsQueue.getIteratorTypeByCommand(command)}:${iterator}`
  }
}
