const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'AdfsSso'

module.exports = class AdfsSso extends BaseOpenAPIClient {

  /**
   * Returns a ADFS SSO settings
   * @param bExtenedSettings - If true then return extended settings
   * @returns {Promise<*>} - ADFS SSO settings
   */
  async GetSettings ({
    bExtenedSettings
  }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetSettings`,
      data: {
        bExtenedSettings
      },
      connection
    }, cacheConfig)
    return this.getResult(body)
  }

  /**
   * Returns a ADFS JWKS
   * @returns {Promise<*>} - ADFS Jwks
   */
  async GetJwks (noParams, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetJwks`,
      connection
    }, cacheConfig)
    return this.getResult(body)
  }

  /**
   * Set a ADFS SSO settings
   * @returns {Promise<*>} - ADFS Jwks
   */
  async SetSettings (pAdfsSettings, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.SetSettings`,
      data: {
        pAdfsSettings: {
          ...pAdfsSettings
        }
      },
      connection
    }, cacheConfig)
    return this.getResult(body)
  }

  /**
   * Set a ADFS SSO enabled/disabled
   * bEnabled - enabled/disabled
   */
  async SetAdfsEnabled({ bEnabled }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.SetAdfsEnabled`,
      data: {
        bEnabled
      },
      connection
    }, cacheConfig)
    return this.getResult(body)
  }

  /**
   * Get a ADFS SSO enabled/disabled
   * @returns {Promise<Boolean>} - Is ADFS enabled
   */
  async GetAdfsEnabled (noParams, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetAdfsEnabled`,
      connection
    }, cacheConfig)
    return this.getResult(body)
  }

  /**
   * Unique string, identifying caller user session
   * wstrUserSessionId - Unique string, identifying caller user session
   * @returns {Promise<*>} - wstrNonce - Output string, containing 'Nonce' value of id_token claim
   *                         wstrState - Output string, containing 'State' value of id_token claim
   */
  async CreateNonceAndStateForUserSession ({ wstrUserSessionId }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CreateNonceAndStateForUserSession`,
      data: {
        wstrUserSessionId
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }
}
