const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'CertUtils'

module.exports = class CertUtils extends BaseOpenAPIClient {
  /**
   * Extract certificates attributes from file.
   * Extract certificates attributes from file in human-readable format.
   * If there is more than one certificate (for example, certifiate file format allows this),
   * an array of attributes params is returned for each found certificate accordingly.
   * @param pCertificateFileChunk - binary certificate data
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<*>} - Array of params with certificate attributes
   */
  async GetCertificateAttributes ({
    pCertificateFileChunk
  }, connection, cacheConfig) {
    const normalizedChunk = pCertificateFileChunk?.value ?? pCertificateFileChunk
    const body = await this.baseRequest({
      command: `${className}.GetCertificateAttributes`,
      data: {
        pCertificateFileChunk: normalizedChunk
      },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Generates self-signed cert.
   * Generates root and leaf certs; returns details about leaf cert
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<*>} - Self-signed certificate information
   */
  async GenerateSelfSignedCertificate ({ pParams }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GenerateSelfSignedCertificate`,
      data: {
        pParams
      },
      connection
    }, cacheConfig)
    return this.getResult(body)
  }

  /**
   * Exports certificate essential elements as params.
   * @param pCert – Certificate in any supported format: DER, PEM or PKCS#12
   * @param pwchPassword - Certificate password
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<*>} - certificate attributes
   */
  async Certificate2Params ({
    pCert, pwchPassword
  }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.Certificate2Params`,
      data: {
        pCert, pwchPassword
      },
      connection
    }, cacheConfig)
    return this.getResult(body)
  }
}
