const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'DatabaseInfo'

module.exports = class DatabaseInfo extends BaseOpenAPIClient {
  /**
   * Get database's files size.
   *
   * @param {*} { args } empty
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {number} size of files of database.
   */
  async GetDBSize ({ args }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetDBSize`,
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Get database's data size.
   *
   * @param {*} { args } empty
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {number} size of data of database.
   */
  async GetDBDataSize ({ args }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetDBDataSize`,
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Get database's events count.
   *
   * @param {*} { args } empty
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {number} count of events of database.
   */
  async GetDBEventsCount ({ args }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetDBEventsCount`,
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Check the server administration and SQL-server permissions to read and write files along path.
   *
   * @param {*} { szwPath }  full-path to checkable directory
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {boolean}
   */
  async CheckBackupPath ({ szwPath }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CheckBackupPath`,
      data: {
        szwPath
      },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Check the server administration and SQL-server permissions to read and write files along path.
   * @param {object} params
   * @param params.szwWinPath  full-path to checkable directory for KSC-server
   * @param params.szwLinuxPath  full-path to checkable directory for SQL-server
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {boolean}
   */
  async CheckBackupPath2 ({ szwWinPath, szwLinuxPath }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CheckBackupPath2`,
      data: {
        szwWinPath,
        szwLinuxPath
      },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Check the server administration and Cloud SQL (Amazon RDS or Azure SQL) permissions to read and write files along Cloud Storage (Amazon S3 or Azure Blob Storage)
   * @param {object} params
   * @param params.nCloudType Cloud type (KLCLOUD::CloudType)
   * @param params.szwCloudPath Checkable cloud (Amazon S3 or Azure Blob Storage) path
   * @param params.szwClientId Client ID for checkable cloud storage (Amazon S3 or Azure Blob Storage)
   * @param params.pSecretChunk Protected utf8 cloud storage (Amazon S3 or Azure Blob Storage) secret
   * @param params.szwStorageKey Azure storage key (empty for Amazon)
   * @param params.szwAzureResName Azure SQL resource name (empty for Amazon)
   * @param params.szwAzureResGroup  Azure SQL resource group (empty for Amazon)
   * @param params.szwAzureAppID Azure AppID (empty for Amazon)
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {boolean}
   */
  async CheckBackupCloudPath ({
    nCloudType,
    szwCloudPath,
    szwClientId,
    pSecretChunk,
    szwStorageKey,
    szwAzureResName,
    szwAzureResGroup,
    szwAzureAppID
  }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CheckBackupCloudPath`,
      data: {
        nCloudType,
        szwCloudPath,
        szwClientId,
        pSecretChunk,
        szwStorageKey,
        szwAzureResName,
        szwAzureResGroup,
        szwAzureAppID
      },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Check the server administration and Cloud SQL (Amazon RDS or Azure SQL) permissions to read and write files along Cloud Storage (Amazon S3 or Azure Blob Storage)
   * @param {object} params
   * @param params.nCloudType Cloud type (KLCLOUD::CloudType)
   * @param params.szwCloudPath Checkable cloud (Amazon S3 or Azure Blob Storage) path
   * @param params.szwClientId Client ID for checkable cloud storage (Amazon S3 or Azure Blob Storage)
   * @param params.pSecretChunk Protected utf8 cloud storage (Amazon S3 or Azure Blob Storage) secret
   * @param params.pStorageKey Protected utf8 Azure storage key (empty for Amazon)
   * @param params.szwAzureResName Azure SQL resource name (empty for Amazon)
   * @param params.szwAzureResGroup  Azure SQL resource group (empty for Amazon)
   * @param params.szwAzureAppID Azure AppID (empty for Amazon)
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {boolean}
   */
  async CheckBackupCloudPath2 ({
    nCloudType,
    szwCloudPath,
    szwClientId,
    pSecretChunk,
    pStorageKey,
    szwAzureResName,
    szwAzureResGroup,
    szwAzureAppID
  }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CheckBackupCloudPath2`,
      data: {
        nCloudType,
        szwCloudPath,
        szwClientId,
        pSecretChunk,
        pStorageKey,
        szwAzureResName,
        szwAzureResGroup,
        szwAzureAppID
      },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Check is current SQL server in cloud (Amazon RDS or Azure SQL)
   *
   * @param { nCloudType } Cloud type (KLCLOUD::CloudType)
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {boolean}
   */
  async IsCloudSQL ({ nCloudType }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.IsCloudSQL`,
      data: {
        nCloudType
      },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  /**
   * Check is current SQL server in on Linux.
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {boolean}
   */
  async IsLinuxSQL ({ args }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.IsLinuxSQL`,
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }
}
