const baseIterator = require('../base-iterator')
const className = 'EventProcessing'

module.exports = class EventProcessing extends baseIterator {
  /**
   * Get record count
   * @param {object} params
   * @param {string} params.strIteratorId - iterator identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  GetRecordCount ({ strIteratorId }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.GetRecordCount`, data: { strIteratorId }, connection }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Returns elements contained in the specified result-set in the diapason from position nStart to position nEnd
   * @param {object} params
   * @param {string} params.strIteratorId - iterator identifier
   * @param {number} params.nStart - start position
   * @param {number} params.nEnd - end position
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  GetRecordRange ({ strIteratorId, nStart, nEnd }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.GetRecordRange`, data: { strIteratorId, nStart, nEnd }, connection }, cacheConfig)
      .then(body => this.deserialize(body, 'pParamsEvents'))
  }

  /**
   * @param {object} params
   * @param {string} params.strIteratorId - iterator identifier
   * @param {Array<object>} [params.events] - contains range blocks to mass delete events
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   * @constructor
   */
  InitiateDelete ({ strIteratorId, pSettings }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.InitiateDelete`, data: { strIteratorId, pSettings: this.serialize(pSettings) }, connection }, cacheConfig)
      .then(body => this.deserialize(body, 'pParamsEvents'))
  }

  /**
   * Закрытие итератора
   * @param {object} params
   * @param {string} params.strIteratorId - идентификатор итератора
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  async ReleaseIterator ({ strIteratorId }, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.ReleaseIterator`,
      data: {
        strIteratorId
      },
      connection
    }, cacheConfig)
    return this.getBody(result)
  }

  /**
   * Получение (с помощью итератора) событий, опубликованных на конкретнеом хосте
   * @param {object} params
   * @param {object} params.iteratorParams
   * @param {string} params.iteratorParams.strDomainName - домен устройства (KLHST_WKS_WINDOMAIN)
   * @param {string} params.iteratorParams.strHostName - имя устройства (KLHST_WKS_HOSTNAME)
   * @param {string} params.iteratorParams.strProduct - идентификатор продукта
   * @param {string} params.iteratorParams.strVersion - версия продукта
   * @param {Array<string>} params.iteratorParams.vecFieldsToReturn - список требуемых полей (List of event attributes)
   * @param {Connection} connection
   * @return {Promise.<*>}
   */
  async fetchEventsForHost ({
                 iteratorParams: {
                   strDomainName,
                   strHostName,
                   strProduct,
                   strVersion,
                   vecFieldsToReturn
                 }
               }, connection) {
    const openAPIClient = require('../open-api-client')
    const fetchFunction = this.getFetchInstance({
      iteratorIdPropertyName: 'strIteratorId',
      resetIterator: openAPIClient.EventProcessingFactory.CreateEventProcessingForHost2.bind(this),
      getRecordCount: this.GetRecordCount.bind(this),
      getRecordRange: this.GetRecordRange.bind(this),
      releaseIterator: this.ReleaseIterator.bind(this)
    })
    const result = await fetchFunction({
      iteratorParams: {
        strDomainName,
        strHostName,
        strProduct,
        strVersion,
        vecFieldsToReturn
      }
    }, connection)
    return result
  }
}
