const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'ExtAud'

/**
 * Provide an interface for working with ExtAudit subsystem.
 * This interface allow you to get a revision of an object and update description.
 */
module.exports = class ExtAud extends BaseOpenAPIClient {
  /**
   * @description Get object revision.
   * @param {object} params
   * @param {bigint} params.nObjId - object id
   * @param {number} params.nObjType - one of object types
   * possible object types:
   * 0 - None
   * 1 - Policy
   * 2 - Task
   * 3 - Package
   * 4 - Server
   * 5 - Virtual server
   * 6 - User
   * 7 - Security Group
   * 8 - Administration Group
   * @param {number} params.nObjRevision - object revision
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetRevision ({ nObjId, nObjType, nObjRevision } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetRevision`,
      data: { nObjId: 'long' + nObjId, nObjType, nObjRevision },
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'pObjParams'))
  }
  /**
   * @description Update revision description.
   * @param {object} params
   * @param {bigint} params.nObjId - object id
   * @param {number} params.nObjType - one of object types
   * possible object types:
   * 0 - None
   * 1 - Policy
   * 2 - Task
   * 3 - Package
   * 4 - Server
   * 5 - Virtual server
   * 6 - User
   * 7 - Security Group
   * 8 - Administration Group
   * @param {number} params.nObjRevision - object revision
   * @param {string} params.wstrNewDescription - new description of revision
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  UpdateRevisionDesc ({ nObjId, nObjType, nObjRevision, wstrNewDescription } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateRevisionDesc`,
      data: { nObjId: 'long' + nObjId, nObjType, nObjRevision, wstrNewDescription },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
  * @description Final delete for deleted objects.
  * @param {object} params
  * @param {Array<Object>} params.arrObjects - Array of pairs ObjId-ObjType. Max size of array is 100 elements.
  * @param {Connection} connection
  * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
  * @return {Promise.<object>}
  */
 FinalDelete ({ arrObjects } = {}, connection, cacheConfig) {
   return this.baseRequest({
     command: `${className}.FinalDelete`,
     data: { arrObjects },
     connection
   }, cacheConfig).then(this.getResult)
 }
}
