const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'FileCategorizer2'

module.exports = class FileCategorizer2 extends BaseOpenAPIClient {

   /**
    * @typedef {object} PCategory
    * @property {string} CategoryType
    * @property {string} name
    * @property {string} uuid
    * @property {string} version
    * @property {boolean} fromMaster
    * @property {number} lastUpdateTime
    * @property {Array} inclusions
    *
    * Returns an object with category attributes like name, description, last update
    * @param {any} nCategoryId - id of category, not guid, ex.: 1
    * @param {Connection} connection 
    * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
    * @returns {PCategory}
    */
    async GetCategory (nCategoryId, connection, cacheConfig) {
        const body = await this.baseRequest({
        command: `${className}.GetCategory`,
        data: {
            nCategoryId
        },
        connection
        }, cacheConfig)
        return this.getBody(body)
    }

    /**
     * Returns array of policies with references to specified category.
     *
     * @param {any} nCatId  - id of category, not guid, ex.: 1
     * @param {Connection} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns Policies array
     */
    async GetRefPolicies (nCatId, connection, cacheConfig) {
        const body = await this.baseRequest({
        command: `${className}.GetRefPolicies`,
        data: {
            nCatId
        },
        connection
        }, cacheConfig)
        return this.getBody(body)
    }

     /**
     * Updating category whith specified ID
     * @param {object} params
     * @param {number} params.nCategoryId Category ID which should be updated
     * @param {object} params.pCategory Params object, can contain only fields that to be updated, ex: {name: 'Foo'}
     * @param {Connection} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns pCategory New category body 
     */
    async UpdateCategory ({ nCategoryId, pCategory }, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.UpdateCategory`,
            data: {
                nCategoryId,
                pCategory
            },
            connection
        }, cacheConfig)
        return this.getBody(body)
    }	

    /**
     * Create category (simple, autoupdate or silverimage)
     * @param {object} params
     * @param {object} params.pCategory Params object, Category body
     * @param {Connection} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns Category id
     */
    async CreateCategory ({ pCategory }, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.CreateCategory`,
            data: {
                pCategory
            },
            connection
        }, cacheConfig)
        return this.deserialize(body, 'PxgRetVal')
    }	


    /**
     * Deleting category with specified nCategoryId
     * 
     * @param {any} nCategoryId 
     * @param {Connection} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns 
     */
    async DeleteCategory (nCategoryId, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.DeleteCategory`,
            data: {
                nCategoryId
            },
            connection
        }, cacheConfig)
        return this.getBody(body)
    }

    
    /**
     * Forces the update Force category update (for autoupdate and silverimage)
     * @param {number} nCategoryId Category id
     * @param {Connection} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns response
     */
    async ForceCategoryUpdate (nCategoryId, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.ForceCategoryUpdate`,
            data: {
                nCategoryId   
            },
            connection
        }, cacheConfig)
        return this.getBody(body)
    }

    /**
     * Initializing url for further file upload by putMethod
     * @param {*} args empty args
     * @param {*} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns {string} URL for uploading
     */
    async InitFileUpload (args, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.InitFileUpload`,
            connection
        }, cacheConfig)
        return this.getBody(body)
    }
    /**
     * 
     * Cancelling created url for file upload
     * @param {any} args 
     * @param {Connection} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns empy
     */
    async CancelFileUpload (args, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.CancelFileUpload`,
            connection
        }, cacheConfig)
        return this.getBody(body)
    }

    /**
     * Get files metadata from zip-archive.
     * To get result use AsyncActionStateChecker.CheckActionState.
     * @param {number} nCategoryId Category id
     * @param {Connection} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns response
     */
    async GetFilesMetadata (ulFlag, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.GetFilesMetadata`,
            data: {
                ulFlag   
            },
            connection
        }, cacheConfig)
        return this.getBody(body)
    }

    /**
     * Get file metadata.
     * To get result use AsyncActionStateChecker.CheckActionState.
     * @param {number} nCategoryId Category id
     * @param {Connection} connection
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns response
     */
    async GetFileMetadata (ulFlag, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.GetFileMetadata`,
            data: {
                ulFlag   
            },
            connection
        }, cacheConfig)
        return this.getBody(body)
    }

    /**
     * Get files metadata from MSI.
     * To get result use AsyncActionStateChecker.CheckActionState.
     * @param {number} nCategoryId Category id
     * @param {Connection} connection 
     * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
     * @returns response
     */
    async GetFilesMetadataFromMSI (ulFlag, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.GetFilesMetadataFromMSI`,
            data: {
                ulFlag   
            },
            connection
        }, cacheConfig)
        return this.getBody(body)
    }





}
