const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'Gcm'

module.exports = class Gcm extends BaseOpenAPIClient {
  /**
   * Retrieves GCM (Google Cloud Messaging) server settings.
   * @param {object} params
   * @param {string} params.wstrSenderId - SENDER_ID (project number), can be retrieved from Google Developers Console.
   * @param {string} params.wstrApiKey - API key, can be retrieved from Google Developers Console.
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>} - True if settings exists; False otherwise.
   */
  GetGcmServerSettings ({ wstrSenderId, wstrApiKey }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetGcmServerSettings`,
      data: { wstrSenderId, wstrApiKey },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Updates GCM (Google Cloud Messaging) server settings.
   * Note: Throws STDE_BADPARAM if wstrSenderId or wstrApiKey empty.
   * @param {object} params
   * @param {string} params.wstrSenderId - SENDER_ID (project number), can be retrieved from Google Developers Console. Can't be empty.
   * @param {string} params.wstrApiKey - API key, can be retrieved from Google Developers Console. Can't be empty.
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>} - True if settings was updated; False otherwise.
   */
  UpdateGcmServerSettings ({ wstrSenderId, wstrApiKey }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateGcmServerSettings`,
      data: { wstrSenderId, wstrApiKey },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Deletes GCM (Google Cloud Messaging) server settings.
   * @param {null} _ - Null-argument, not used in method
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  DeleteGcmServerSettings (_, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteGcmServerSettings`,
      data: null,
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Retrieves GCM settings propagation option. GCM settings can be propagated to virtual server in one case only - if it is absent on virtual server.
   * Note: This option can be retrieved from main server only.
   * @param {object} params
   * @param {boolean} params.bPropagate2VS - determines if GCM settings should be proragated from main server to virtual servers.
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  GetGcmPropagation2VS ({ bPropagate2VS }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetGcmPropagation2VS`,
      data: { bPropagate2VS },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Sets possibility for GCM settings propagation from main server to virtual servers. GCM settings can be propagated to virtual server in one case only - if it is absent on virtual server.
   * Note: This option can be set from main server only.
   * @param {object} params
   * @param {boolean} params.bPropagate2VS - determines if GCM settings should be proragated from main server to virtual servers.
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  SetGcmPropagation2VS ({ bPropagate2VS }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetGcmPropagation2VS`,
      data: { bPropagate2VS },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * It checks if GCM server settings are present.
   * Note: It takes in account gcm propagation option.
   * @param {null} _ - Null-argument, not used in method
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>} - True if settings are present; False otherwise.
   */
  CheckIfGcmServerSettingsPresent (_, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CheckIfGcmServerSettingsPresent`,
      data: null,
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * It checks if GCM server settings should be set.
   * Note: It takes in account gcm propagation option.
   * @param {null} _ - Null-argument, not used in method
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>} - True if server settings should be set; False otherwise.
   */
  CheckIfGcmServerSettingsShouldBeSet (_, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CheckIfGcmServerSettingsShouldBeSet`,
      data: null,
      connection
    }, cacheConfig).then(this.getResult)
  }
}
