const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'GroupSync'

/**
 * Access to group synchronization objects
 */
module.exports = class GroupSync extends BaseOpenAPIClient {
  /**
   * Acquire group synchronization properties.
   * Returns requested properties of the specified group synchronization
   * @param {object} params
   * @param {number} params.nSync - id of the group synchronization
   * @param {Array<string>} params.arrFieldsToReturn - array of attribute names to return
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns Promise<object> - container with values of required attributes
   */
  GetSyncInfo ({ nSync = 0, arrFieldsToReturn = [] }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetSyncInfo`,
      data: this.serialize({
        nSync,
        arrFieldsToReturn
      }),
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Acquire group synchronization delivery time for the specified host.
   * Returns UTC time when the specified synchronization has been delivered to the specified host
   * @param {object} params
   * @param {number} params.nSync - id of the group synchronization
   * @param {string} params.szwHostId - host name (see KLHST_WKS_HOSTNAME)
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async GetSyncDeliveryTime ({ nSync, szwHostId }, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.GetSyncDeliveryTime`,
      data: {
        nSync,
        szwHostId
      },
      connection
    }, cacheConfig)
    return this.getResult(result)
  }

  /**
   * Acquire group synchronization state at target hosts.
   * @param {object} params
   * @param {number} params.nSync - id of the group synchronization
   * @param {Array<string>} params.arrFieldsToReturn - array of attribute names to return
   * @param {Array<Object} params.arrFieldsToOrder - array of containers for sorting {Name: string, Asc: bool}
   * @param {number} params.nLifeTime - timeout in seconds to keep the result-set alive, zero means 'default value'
   * @param {*} connection 
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns Promise<string> - iterator id
   */
  async GetSyncHostsInfo ({ nSync, arrFieldsToReturn, arrFieldsToOrder, nLifeTime }, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.GetSyncHostsInfo`,
      data: this.serialize({
        nSync,
        arrFieldsToReturn,
        arrFieldsToOrder,
        nLifeTime
      }),
      connection
    }, cacheConfig)
    return this.getResult(result)
  }
}
