const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'GroupTaskControlApi'

module.exports = class GroupTaskControlApi extends BaseOpenAPIClient {

  /**
   * Returns all task data for a group/set task with a given object identity and revision.
   * @param {object} params
   * @param {number} params.nObjId Task identity
   * @param {number} params.nRevision Task revision id
   * @param {Connection} connection 
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns list of attributes of executable file
   */
    async GetTaskByRevision ({nObjId, nRevision}, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.GetTaskByRevision`,
            data: {
                nObjId,
                nRevision
            },
            connection
        }, cacheConfig)
        return this.getBody(body)
    }

   /**
   * Request statistics of the given tasks.
   * @param {object} params
   * @param {Array<number>} params.pTasksIds Task identity
   * @param {Connection} connection 
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns list of attributes of executable file
   */
  async RequestStatistics ({pTasksIds}, connection, cacheConfig) {
        const body = await this.baseRequest({
            command: `${className}.RequestStatistics`,
            data: {
                pTasksIds
            },
            connection
        }, cacheConfig)
        return this.getBody(body)
    }

   /**
   * Restore task from revision.
   * Rolls back the group/set task specified by nObjId to the revision nRevision.
   * @param {object} params
   * @param {number} params.nObjId Task identity
   * @param {number} params.nRevision Task revision id
   * @param {Connection} connection 
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns list of attributes of executable file
   */
  async RestoreTaskFromRevision ({nObjId, nRevision}, connection, cacheConfig) {
    const body = await this.baseRequest({
        command: `${className}.RestoreTaskFromRevision`,
        data: {
            nObjId,
            nRevision
        },
        connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Exports task.
   * Gets specific task by its identifier and save data to memory chunk. Chunk can be later saved to file or sent over network
   * @param {object} params
   * @param {number} params.wstrTaskId Task identifier 
   * @param {Connection} connection 
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns Pointer to memory chunk with exported task data 
   */
   async ExportTask ({ wstrTaskId }, connection, cacheConfig) {
    const body = await this.baseRequest({
        command: `${className}.ExportTask`,
        data: {
            wstrTaskId
        },
        connection
    }, cacheConfig)
    return this.getResult(body)
  }

  /**
   * Prepares task import operation. 
   * This method does not perform actual import of task: instead, it prepares import operation by saving task blob and associating unique identifier with it, 
   * which should be later used in GroupTaskControlApi::CommitImportedTask method. 
   * Some tasks have import security restrictions, thus import of task can be allowed or denied. 
   * To determine these restrictions, one should analyze info, returned via output parameter pCommitInfo (see detailed parameter descriptions in parameter section), 
   * and pass analyze result in bCommit parameter of GroupTaskControlApi::CommitImportedTask.
   * @param {object} params
   * @param {number} params.pBlob Pointer to memory chunk with imported task data 
   * @param {object} params.pExtraData Task import options
   * @param {object} params.pCommitInfo Output commit info , used to decide whether we can continue importing task with GroupTaskControlApi::CommitImportedTask method, or not
   * @param {Connection} connection 
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns Unique identifier of task blob for later import
   */
   async ImportTask ({ pBlob, pExtraData, skipSerialization }, connection, cacheConfig) {
    const extraData = skipSerialization ? pExtraData : this.serialize(pExtraData)

    const body = await this.baseRequest({
        command: `${className}.ImportTask`,
        data: {
            pBlob,
            pExtraData: extraData
        },
        connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Completes import of task.
   * Performs actual import of task, which was initiated earlier by GroupTaskControlApi::ImportTask method.
   * This method works with task data, which was already processed and analyzed. 
   * At time of calling this method, one should have appropriate rights for task creation. 
   * In case of group task, there should be write access to that group. 
   * If imported task is task for specified hosts, there should be read access to that hosts. 
   * If task is query-based task, there should be read access to related query. 
   * For additional info on task import options, see pExtraData parameter of GroupTaskControlApi::ImportTask method.
   * @param {object} params
   * @param {string} params.wstrId Identifier of blob with task data. Use GroupTaskControlApi::ImportTask method to get its value
   * @param {boolean} params.bCommit Whether to commit or not. If bCommit is true, method will perform its work and complete import operation by creating new task and returning its identifier. If bCommit is false, method will cleanup old data and import will be cancelled. 
   * @param {Connection} connection 
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns Unique identifier of new task if bCommit was set to true, and empty string otherwise 
   */
   async CommitImportedTask ({ wstrId, bCommit }, connection, cacheConfig) {
    const body = await this.baseRequest({
        command: `${className}.CommitImportedTask`,
        data: {
            wstrId,
            bCommit
        },
        connection
    }, cacheConfig)
    return this.getResult(body)
  }
}
