const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'HostGroup'

module.exports = class HostGroup extends BaseOpenAPIClient {
  /**
   * Methods for memoization
   * @returns {[]}
   */
  get memoizeMethods () {
    return ['GetStaticInfo']
  }

  /**
   * Получение идентификатора предопределённой группы "Managed computers"
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  GroupIdGroups (noParams, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GroupIdGroups`,
      data: {},
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Получение идентификатора предопределённой группы "Unassigned computers"
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  GroupIdUnassigned (noParams, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GroupIdUnassigned`,
      data: {},
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Get id of predefined group "Master server".
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async GroupIdSuper (noParams, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.GroupIdSuper`,
      data: {},
      connection
    }, cacheConfig)
    const groupIdSuper = this.getResult(result)
    return groupIdSuper
  }

  /**
   * Returns Administration Group info
   * @deprecated Use GetGroupInfoEx instead
   *
   * @param { object } props - Method props
   * @param { number } props.nGroupId - Administration Group ID
   * @param { boolean } props.deserialize - If true, deserializes result
   * @param {Connection} connection - OpenApi connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async GetGroupInfo ({ nGroupId, deserialize = false }, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.GetGroupInfo`,
      data: { nGroupId },
      connection
    }, cacheConfig)
    const groupInfo = deserialize ? this.deserialize(result) : this.getResult(result)
    return groupInfo
  }

  /**
   * Returns Administration Group info
   *
   * @param { object } props - Method props
   * @param { number } props.nGroupId - Administration Group ID
   * @param { string[] } props.pArrAttributes - List of Group fields to return
   * @param {Connection} connection - OpenApi connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async GetGroupInfoEx ({ nGroupId, pArrAttributes }, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.GetGroupInfoEx`,
      data: {
        nGroupId,
        pArrAttributes
      },
      connection
    }, cacheConfig)
    return this.deserialize(result)
  }

  /**
   * Поиск хостов с помощью фильтров
   * @param {object} params
   * @param {string} params.wstrFilter - фильтр, содеражищий условия по атрибутам хоста
   * @param {string[]} params.vecFieldsToReturn - список названий атрибутов хоста, которые должны быть возвращены
   * @param {Object[]} params.vecFieldsToOrder - список контейнеров (объектов) с настройками сортировки
   * @param {object} params.pParams - дополнительные праметры
   * @param {number} params.lMaxLifeTime - максимальное время жизни итератора в секундах, не более чем 7200
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  FindHosts ({ wstrFilter, vecFieldsToReturn, vecFieldsToOrder = [], pParams = null, lMaxLifeTime = 7200, skipDeserialization = false } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.FindHosts`,
      data: { wstrFilter, vecFieldsToReturn, vecFieldsToOrder, pParams, lMaxLifeTime },
      connection
    }, cacheConfig)
      .then(body => skipDeserialization ? body.body : this.deserialize(body, 'strAccessor'))
  }

  /**
   * Перемещение нескольких хостов в группу администрирования
   * @param {object} params
   * @param {number} params.nGroup - идентификатор группы назначения
   * @param {Array<String>} params.pHostNames - список имён хостов (KLHST_WKS_HOSTNAME)
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  MoveHostsToGroup ({ nGroup, pHostNames } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.MoveHostsToGroup`,
      data: { nGroup, pHostNames },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Удаление нескольких хостов
   * @param {object} params
   * @param {Array<String>} params.pHostNames - имена удаляемых хостов (KLHST_WKS_HOSTNAME)
   * @param {boolean} params.bForceDestroy - true: хост будет удалён; false: управляемое устройство будет перемещено в Unassigned
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  RemoveHosts ({ pHostNames, bForceDestroy = false } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RemoveHosts`,
      data: { pHostNames, bForceDestroy },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  GetSubgroups (params, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetSubgroups`,
      data: params,
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }
  AddGroup (params, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddGroup`,
      data: params,
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }
  UpdateGroup ({ nGroup, pInfo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateGroup`,
      data: {
        nGroup,
        pInfo: this.serialize(pInfo)
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }
  RemoveGroup ({ nGroup, nFlags = 1 }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RemoveGroup`,
      data: { nGroup, nFlags },
      connection
    }, cacheConfig)
      .then(result => this.getResult(result, 'strActionGuid'))
  }
  ResolveAndMoveToGroup ({ pInfo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ResolveAndMoveToGroup`,
      data: { pInfo },
      connection
    }, cacheConfig).then(result => this.getResult(result, 'pResults'))
  }

  /**
   * Finds groups that satisfy conditions from filter string wstrFilter, and creates a server-side collection of found groups
   * @param {object} params
   * @param {string} params.wstrFilter - filter string, contains a condition over List of group attributes
   * @param {Array} params.vecFieldsToReturn - array of group attribute names to return. See List of group attributes for attribute names
   * @param {Array} params.vecFieldsToOrder -  array of containers each of them containing two attributes
   * @param {object} params.pParams - extra options
   * @param {number} params.lMaxLifeTime - max result-set lifetime in seconds, not more than 7200
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async FindGroups ({ wstrFilter, vecFieldsToReturn = [], vecFieldsToOrder = [], pParams = null, lMaxLifeTime = 7200 }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.FindGroups`,
      data: {
        wstrFilter,
        vecFieldsToReturn,
        vecFieldsToOrder,
        pParams,
        lMaxLifeTime
      },
      connection
    }, cacheConfig)
    const result = this.getResult(body, 'strAccessor')
    return result
  }

  GetStaticInfo (params, connection, cacheConfig) {
    // pValues
    return this.baseRequest({
      command: `${className}.GetStaticInfo`,
      data: {
        pValues: params
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  GetRunTimeInfo (params, connection, cacheConfig) {
    // pValues
    return this.baseRequest({
      command: `${className}.GetRunTimeInfo`,
      data: {
        pValues: params
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Получение информации о хосте
   * @param {object} params
   * @param {string} params.strHostName - параметр хоста KLHST_WKS_HOSTNAME
   * @param {Array<string>} params.pFields2Return - список полей, которые должны быть возвращены
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  GetHostInfo ({ strHostName, pFields2Return, deserialize = false }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetHostInfo`,
      data: {
        strHostName,
        pFields2Return
      },
      connection
    }, cacheConfig).then(deserialize ? this.deserialize : this.getResult)
  }

  /**
   * Обновление (сохранение) информации о хосте
   * @param {object} params
   * @param {string} params.strHostName - имя хоста (KLHST_WKS_HOSTNAME)
   * @param {object} params.pInfo - контейнер с атрибутами хоста, которые должны быть обновлены
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  UpdateHost ({ strHostName, pInfo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateHost`,
      data: {
        strHostName,
        pInfo
      },
      connection
    }, cacheConfig).then(this.getBody)
  }
  // eslint-disable-next-line camelcase
  SS_GetNames ({ strHostName, strType, strProduct, strVersion }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SS_GetNames`,
      data: {
        strHostName, strType, strProduct, strVersion
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Чтение из хранилища настроек (settings storage) устройства
   * @param {object} params
   * @param {string} params.strHostName - имя устройства
   * @param {string} params.strType - тип хранилища
   * @param {string} params.strProduct - идентификатор продукта
   * @param {string} params.strVersion - версия продукта
   * @param {string} params.strSection - секция хранилища
   * @param {boolean} params.skipDeserialization - флаг, нужно ли десериализовывать ответ
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  // eslint-disable-next-line camelcase
  SS_Read ({ strHostName, strType = 'SS_SETTINGS', strProduct, strVersion, strSection, skipDeserialization = false }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SS_Read`,
      data: {
        strHostName, strType, strProduct, strVersion, strSection
      },
      connection
    }, cacheConfig).then(skipDeserialization ? this.getBody : this.deserialize)
  }
  // eslint-disable-next-line camelcase
  SS_Write ({ strHostName, strType, strProduct, strVersion, strSection, nOption, pSettings, skipSerialization }, connection, cacheConfig) {
    let _pSettings = skipSerialization ? pSettings : this.serialize(pSettings)
    return this.baseRequest({
      command: `${className}.SS_Write`,
      data: {
        strHostName, strType, strProduct, strVersion, strSection, nOption, pSettings: _pSettings
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Delete section from host settings storage.
   * @param strHostName - host name (unique server-generated string)
   * @param strType - type of storage (for example: "SS_SETTINGS")
   * @param strProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param strVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param strSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   * @constructor
   */
  // eslint-disable-next-line camelcase
  SS_DeleteSection ({ strHostName, strType = 'SS_SETTINGS', strProduct, strVersion, strSection }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SS_DeleteSection`,
      data: {
        strHostName, strType, strProduct, strVersion, strSection
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Create section in host settings storage
   * @param strHostName - host name (unique server-generated string)
   * @param strType - type of storage (for example: "SS_SETTINGS")
   * @param strProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param strVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param strSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   * @constructor
   */
  // eslint-disable-next-line camelcase
  SS_CreateSection ({ strHostName, strType = 'SS_SETTINGS', strProduct, strVersion, strSection }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SS_CreateSection`,
      data: {
        strHostName, strType, strProduct, strVersion, strSection
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Получение информации об установленных на хосте продуктах
   * @param {object} params
   * @param {string} params.strHostName - идентификатор устройства aka KLHST_WKS_HOSTNAME
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  GetHostProducts ({ strHostName }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetHostProducts`,
      data: {
        strHostName
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Поиск пользователей, которые удовлетворяют условиям фильтра (параметр strFilter)
   * @param {object} params
   * @param {string} params.strFilter - фильтр
   * @param {string[]} params.pFieldsToReturn - список атрибутов пользователя, которые должны быть возвращены
   * @param {Object[]} params.pFieldsToOrder - массив контейнеров (объектов), содепржащих настройки сортировки
   * @param {number} params.lMaxLifeTime - максимальное время жизни итератора в секундах, не более чем 7200
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  FindUsers ({ strFilter, pFieldsToReturn, pFieldsToOrder = [], lMaxLifeTime = 7200 } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.FindUsers`,
      data: { strFilter, pFieldsToReturn, pFieldsToOrder, lMaxLifeTime },
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'strAccessor'))
  }

  /**
   * Получение серверного идентификатора для управления задачми устройства
   * @param {object} params
   * @param {string} params.strHostName - имя хоста (KLHST_WKS_HOSTNAME)
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<{result}>}
   * @constructor
   */
  GetHostTasks ({ strHostName }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetHostTasks`,
      data: { strHostName },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Обнуление количества найденных вирусов на хостах
   * @param {object} params
   * @param {Array<string>} params.pHostNames - список имён хостов, у которых нужно обнулить количество найденных вирусов
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  ZeroVirusCountForHosts ({ pHostNames }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ZeroVirusCountForHosts`,
      data: { pHostNames },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Получение списка установленных для продукта патчей
   * @param {object} params
   * @param {string} params.strHostName - имя устройства (KLHST_WKS_HOSTNAME)
   * @param {string} params.strProductName - идентификатор продукта
   * @param {string} params.strProductVersion - версия продукта
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<{result}|*>}
   */
  async GetHostfixesForProductOnHost ({ strHostName, strProductName, strProductVersion }, connection, cacheConfig) {
    const installedPatchesResponse = await this.baseRequest({
      command: `${className}.GetHostfixesForProductOnHost`,
      data: { strHostName, strProductName, strProductVersion },
      connection
    }, cacheConfig)
    return this.deserialize(installedPatchesResponse)
  }

  RestartNetworkScanning ({ nType }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RestartNetworkScanning`,
      data: { nType },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Acquire administration group id by its name and id of parent group.
   * @param nParent - Id of parent group
   * @param strName - name of group
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<any>}
   */
  GetGroupId ({ nParent, strName }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetGroupId`,
      data: { nParent, strName },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * List of Windows domain in the network.
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>}
   */
  GetDomains (noParams, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetDomains`,
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Return a list of workstation names in the domain
   * The information is obtained from the domain controller. This call returns the full list of workstations in the domain, even if the workstation is now turned off
   * @param domain - domain name
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<any>} - array of hosts in domain
   * @constructor
   */
  GetDomainHosts ({ domain }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetDomainHosts`,
      data: { domain },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Finds hosts asynchronously that satisfy conditions from filter string wstrFilter, and creates a server-side collection of found hosts
   * @param wstrFilter - filter string, contains a condition over host attributes
   * @param vecFieldsToReturn - array of host attribute names to return
   * @param vecFieldsToOrder - array of containers each of them containing two attributes "Name" and "Asc"
   * @param pParams - extra options
   * @param lMaxLifeTime - max result-set lifetime in seconds, not more than 7200
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - strRequestId
   */
  FindHostsAsync ({ wstrFilter, vecFieldsToReturn, vecFieldsToOrder = [], pParams, lMaxLifeTime = 7200 }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.FindHostsAsync`,
      data: {
        wstrFilter,
        vecFieldsToReturn,
        vecFieldsToOrder,
        pParams,
        lMaxLifeTime
      },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Get result of FindHostsAsync operation
   * @param strRequestId - identity of asynchronous operation
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} { strAccessor, pFailedSlavesInfo } - result-set ID, identifier of the server-side ordered collection of found hosts. The result-set is destroyed and associated memory is freed in following cases; information about slave servers the search for which failed due to various reasons, contains array KLGRP_FAILED_SLAVES_PARAMS
   */
  FindHostsAsyncGetAccessor ({ strRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.FindHostsAsyncGetAccessor`,
      data: {
        strRequestId
      },
      connection
    }, cacheConfig).then(body => this.getResult(body, 'strAccessor'))
  }

  GetInstanceStatistics ({ vecFilterFields }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetInstanceStatistics`,
      data: {
        vecFilterFields
      },
      connection
    }, cacheConfig).then(body => this.getResult(body, 'PxgRetVal'))
  }

  /**
   * Add hosts to synchronization
   * @param {object} params
   * @param {Array<string>} params.pHostNames - array of host names
   * @param {string} params.strSSType - setting storage identity (empty string means synchronization of all setting storages)
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.string>}
   */
  async AddHostsForSync ({ pHostNames, strSSType = '' }, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.AddHostsForSync`,
      data: {
        pHostNames,
        strSSType
      },
      connection
    }, cacheConfig)
    return this.getResult(result, 'strActionGuid')
  }

  /**
   * Return array of product components for specified host and product.
   * @param {object} params
   * @param {string} params.strHostName - host name
   * @param {string} params.strProductName - product name
   * @param {string} params.strProductVersion - product version
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<{result}>}
   * @constructor
   */
  GetComponentsForProductOnHost ({ strHostName, strProductName, strProductVersion }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetComponentsForProductOnHost`,
      data: { strHostName, strProductName, strProductVersion },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Update attributes of multiple computers.
   * @param {object} params
   * @param {array} params.pArrHostIds - Array of up to 100 strings. Each entry is a host name, a unique server-generated string (see KLHST_WKS_HOSTNAME attribute). It is NOT the same as computer network name (DNS-, FQDN-, NetBIOS-name)
   * @param {object} params.pProperties - container with host attributes to be modified. See List of host attributes. Following properties can be specified: KLHST_MANAGED_OTHER_SERVER
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<{result}>} - Array of failed identifiers. If Administration Server failed to change host attributes it puts host identifier into this array.
   * @constructor
   */
  UpdateHostsMultiple ({ pArrHostIds, pProperties }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateHostsMultiple`,
      data: { pArrHostIds, pProperties },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Find incident by filter string.
   * Finds incidents that satisfy conditions from filter string strFilter.
   * @param {object} params
   * @param {string} params.strFilter - incident filtering expression (see Search filter syntax). See the list of incident attributes that can be used in this expression in Remarks section below
   * @param {array} params.pFieldsToReturn - array of incident attribute names to return. See List of incident attributes for attribute names
   * @param {array} params.pFieldsToOrder - array of containers each of them containing two attributes: -"Name" of type String, name of attribute used for ordering (see Remarks below); -"Asc" of type Boolean, ascending if true descending otherwise;
   * @param {number} params.lMaxLifeTime - max lifetime of accessor (sec)
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<{strAccessor}>} - result-set ID, identifier of the server-side ordered collection of found incidents. The result-set is destroyed and associated memory is freed in following cases:
   * -Passed lMaxLifeTime seconds after last access to the result-set (by methods ChunkAccessor.GetItemsCount and ChunkAccessor.GetItemsChunk
   * -Session to the Administration Server has been closed.
   * -ChunkAccessor.Release has been called.
   * @constructor
   */
  FindIncidents ({ strFilter, pFieldsToReturn, pFieldsToOrder, lMaxLifeTime }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.FindIncidents`,
      data: { strFilter, pFieldsToReturn, pFieldsToOrder, lMaxLifeTime },
      connection
    }, cacheConfig).then(body => this.getResult(body, 'strAccessor'))
  }

  /**
   *Creates new incident.
   * @param {object} params
   * @param {object} params.pData - container with incident attributes Attributes can be set:
   * - "KLINCDT_SEVERITY" (required)
   * - "KLINCDT_ADDED" (required)
   * - "KLINCDT_BODY" (required)
   * - "KLINCDT_IS_HANDLED" (optional, false by default)
   * - "KLHST_WKS_HOSTNAME" (required)
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<number>} - unique incident id
   */
  AddIncident ({ pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddIncident`,
      data: { pData },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Deletes incident
   *Grou
   * @param {object} params
   * @param {number} params.nId - incident id
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  DeleteIncident ({ nId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteIncident`,
      data: { nId },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Modifies properties of an existing incident.
   * @param {object} params
   * @param {number} params.nId - incident id
   * @param {object} params.pData - container with incident attributes Attributes can be set:
   * - "KLINCDT_SEVERITY" (required)
   * - "KLINCDT_ADDED" (required)
   * - "KLINCDT_BODY" (required)
   * - "KLINCDT_IS_HANDLED" (optional, false by default)
   * - "KLHST_WKS_HOSTNAME" (required)
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  UpdateIncident ({ nId, pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateIncident`,
      data: { nId, pData },
      connection
    }, cacheConfig).then(this.deserialize)
  }
}
