const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'HostTasks'

module.exports = class HostTasks extends BaseOpenAPIClient {
  ResetTasksIterator ({ strSrvObjId, strProductName = '', strVersion = '', strComponentName = '', strInstanceId = '', strTaskName = '' }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ResetTasksIterator`,
      data: { strSrvObjId, strProductName, strVersion, strComponentName, strInstanceId, strTaskName },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  AddTask ({ strSrvObjId, taskData, skipSerialization }, connection, cacheConfig) {
    const pData = skipSerialization ? taskData : this.serialize(taskData)
    return this.baseRequest({
      command: `${className}.AddTask`,
      data: { strSrvObjId, pData },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Получение списка задач
   * @param {object} params
   * @param {string} params.strSrvObjId - идентификатор серверного объекта, полученный из метода HostGroup.GetHostTasks
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<{result}>}
   * @constructor
   */
  GetNextTask ({ strSrvObjId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetNextTask`,
      data: { strSrvObjId },
      connection
    }, cacheConfig)
      .then(body => this.deserialize(body, 'pTaskData'))
  }

  /**
   * Getting host local task data
   * @param {object} params
   * @param {string} params.strSrvObjId - server object ID that got from HostGroup.GetHostTasks
   * @param {string} params.strTask - Tasks storage identifier of the task
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {object} task settings
   */
  async GetTaskData ({ strSrvObjId, strTask }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetTaskData`,
      data: { strSrvObjId, strTask },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Modifying host local task
   * @param {object} params
   * @param {string} params.strSrvObjId - server object ID that got from HostGroup.GetHostTasks
   * @param {string} params.strTask - Tasks storage identifier of the task
   * @param {string} params.pData - task settings, see Task settings format
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns
   */
  async UpdateTask ({ strSrvObjId, strTask, pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateTask`,
      data: {
        strSrvObjId,
        strTask,
        pData: this.serialize(pData)
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  GetTaskStartEvent	({ strTask, strSrvObjId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetTaskStartEvent`,
      data: { strSrvObjId, strTask },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  SetTaskStartEvent ({ strSrvObjId, taskId, startEventData, skipSerialization }, connection, cacheConfig) {
    const strTask = taskId && taskId.toString()
    const pData = skipSerialization ? startEventData : this.serialize(startEventData)
    return this.baseRequest({
      command: `${className}.SetTaskStartEvent`,
      data: {
        strTask,
        pData,
        strSrvObjId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Deleting local host task
   * @param {object} params
   * @param {string} params.strSrvObjId - server object ID that got from HostGroup.GetHostTasks
   * @param {string} params.strTask - Tasks storage identifier of the task
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  async DeleteTask ({ strSrvObjId, strTask }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteTask`,
      data: { strSrvObjId, strTask },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Получение списка задач для устройства
   * !важно: это не Open API method, только обёртка над OAPI методами
   * @param {object} params
   * @param {string} params.strHostName - имя хоста (KLHST_WKS_HOSTNAME)
   * @param {Connection} connection
   * @return {Promise.<{result: {tasks: Array}}>}
   */
  async GetHostTasks ({ strHostName }, connection) {
    const strSrvObjId = await this._getStrSrvObjId(strHostName, connection)
    await this.ResetTasksIterator({ strSrvObjId }, connection)
    const tasks = await getHostTaskRecursive.call(this, strSrvObjId, connection)
    return {
      result: {
        tasks
      }
    }

    /**
     * Рекурсивное получение задач устройства
     * Задачи получаются одна за другой
     * @param {string} strSrvObjId - серверный идентификатор для управления задачми устройства
     * @param {Connection} connection
     * @param {Array<Object>} tasks - резщультирующий список полученных задач
     * @return {Promise.<*>}
     */
    async function getHostTaskRecursive (strSrvObjId, connection, tasks = []) {
      const { result: task } = await this.GetNextTask({ strSrvObjId }, connection)
      if (task) {
        tasks.push(task)
        return getHostTaskRecursive.call(this, strSrvObjId, connection, tasks)
      } else {
        return tasks
      }
    }
  }

  /**
   * Получение идентификатора серверного объекта
   * @param {string} strHostName - имя устройства
   * @param {Connection} connection
   * @return {Promise.<object>}
   * @private
   */
  _getStrSrvObjId (strHostName, connection) {
    const openAPIClient = require('../open-api-client')
    return openAPIClient
      .HostGroup
      .GetHostTasks({ strHostName }, connection)
      .then(({ result: strSrvObjId }) => strSrvObjId)
  }
}
