const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'HstAccessControl'

/**
 * Provide user authentication methods.
 */
module.exports = class HstAccessControl extends BaseOpenAPIClient {
  /**
   * @description Add user role. A role can be added only at a main server.
   * @param {object} params
   * @param {string} params.KLHST_ACL_ROLE_DN - role display name
   * @param {object} params.role_products - role access rights object
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  // eslint-disable-next-line
  AddRole ({ KLHST_ACL_ROLE_DN, role_products } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddRole`,
      data: {
        pRoleData: { KLHST_ACL_ROLE_DN, role_products }
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Return parameters of a role.
   * @param {object} params
   * @param {number} params.nId - id of a role
   * @param {Array<string>} params.pFieldsToReturn - array of strings with role attribute names to return
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetRole ({ nId, pFieldsToReturn }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetRole`,
      data: { nId, pFieldsToReturn },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * @description Delete user role
   * @param {object} params
   * @param {number} params.nId - id of a role
   * @param {boolean} params.bProtection - if true then it checks that the user does not reduce rights for himself
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  DeleteRole ({ nId, bProtection = false }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteRole`,
      data: { nId, bProtection },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Update user role
   * @param {object} params
   * @param {number} params.KLHST_ACL_ROLE_ID - id of a role
   * @param {string} params.KLHST_ACL_ROLE_NAME - guid name of a role
   * @param {string} params.displayName - human name of a role
   * @param {object} params.role_products - role access rights object
   * @param {boolean} params.bProtection - if true then it checks that the user does not reduce rights for himself
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  // eslint-disable-next-line
  UpdateRole ({ KLHST_ACL_ROLE_NAME, KLHST_ACL_ROLE_ID, displayName, role_products, bProtection = false }, connection, cacheConfig) {
    let params = {
      nId: KLHST_ACL_ROLE_ID,
      pRoleData: {
        KLHST_ACL_ROLE_NAME,
        KLHST_ACL_ROLE_DN: displayName,
        role_products
      },
      bProtection
    }
    return this.baseRequest({
      command: `${className}.UpdateRole`,
      data: params,
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Find roles by filter string
   * @param {object} params
   * @param {string} params.filters - filter string
   * @param {Array<string>} params.fields - array of strings with role attribute names to return
   * @param {Array<object>} params.sort - array of containers each of them containing two attributes
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  FindRoles ({ filters, fields, sort }, connection, cacheConfig) {
    var params = {
      strFilter: filters,
      pFieldsToReturn: fields,
      pFieldsToOrder: sort,
      lMaxLifeTime: 7200
    }
    return this.baseRequest({
      command: `${className}.FindRoles`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'strAccessor'))
  }

  /**
   * @description Find roles by filter string (extended by iterator)
   * @param {object} params
   * @param {string} params.filters - filter string
   * @param {Array<string>} params.fields - array of strings with role attribute names to return
   * @param {Array<object>} params.sort - array of containers each of them containing two attributes
   * @param {number} params.nStart - start position for counter
   * @param {number} params.nEnd - end position for counter
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  FindRolesExtended ({ filters, fields, sort = [], nStart = 0, nEnd }, connection, cacheConfig) {
    var params = {
      strFilter: filters,
      pFieldsToReturn: fields,
      pFieldsToOrder: sort,
      lMaxLifeTime: 7200
    }
    return this.baseRequest({
      command: `${className}.FindRoles`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'strAccessor'))
      .then(iteratorContainer => {
        const strAccessor = iteratorContainer.result
        const OpenApiClient = require('../open-api-client')
        return OpenApiClient.ChunkAccessor.GetItemsCount({ strAccessor }, connection)
          .then(countContainer => {
            const count = countContainer.result
            const params = {
              strAccessor,
              nStart,
              nCount: nEnd || count
            }
            return OpenApiClient.ChunkAccessor.GetItemsChunk(params, connection)
          })
          .then(responseContainer => {
            return OpenApiClient.ChunkAccessor.Release({ strAccessor }, connection)
              .then(() => responseContainer)
          })
      })
  }

  /**
   * @description Returns ACL for the specified object.
   * @param {object} params
   * @param {number} params.nObjId - object id
   * @param {number} params.nObjType - object type
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetScObjectAcl ({ nObjId, nObjType = 1 }, connection, cacheConfig) {
    var params = { nObjId, nObjType }
    return this.baseRequest({
      command: `${className}.GetScObjectAcl`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.deserialize(body, 'pAclParams'))
  }

  /**
   * @description Sets ACL for the specified object.
   * @param {object} params
   * @param {number} params.nObjId - object id
   * @param {number} params.nObjType - object type
   * @param {object} params.pAclParams - acl
   * @param {boolean} params.bCheckCurrentUserAce - if true then it checks that the user does not reduce rights for himself
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  SetScObjectAcl ({ nObjId, nObjType = 1, pAclParams, bCheckCurrentUserAce = false }, connection, cacheConfig) {
    var params = { nObjId, nObjType, pAclParams: this.serialize(pAclParams), bCheckCurrentUserAce }
    return this.baseRequest({
      command: `${className}.SetScObjectAcl`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.deserialize(body, 'pAclParams'))
  }

  /**
   * @description Returns descriptions of visual view for access rights in KSC.
   * @param {object} params
   * @param {string} params.wstrLangCode - IETF language tag (for example: en-us)
   * @param {number} params.nObjId - object id
   * @param {number} params.nObjType - object type
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetVisualViewForAccessRights ({ wstrLangCode, nObjId, nObjType = 1 }, connection, cacheConfig) {
    let params = { wstrLangCode, nObjId, nObjType }
    return this.baseRequest({
      command: `${className}.GetVisualViewForAccessRights`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.deserialize(body, 'pViewParams'))
  }

  /**
   * @description Searches for trustees meeting specified criteria
   * @param {object} params
   * @param {string} params.filters - filter string
   * @param {Array<string>} params.fields - array of strings with role attribute names to return
   * @param {Array<object>} params.sort - array of containers each of them containing two attributes
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  FindTrustees ({ filters, fields, sort = [] }, connection, cacheConfig) {
    var params = {
      strFilter: filters,
      pFieldsToReturn: fields,
      pFieldsToOrder: sort,
      lMaxLifeTime: 7200
    }
    return this.baseRequest({
      command: `${className}.FindTrustees`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'strAccessor'))
  }

  /**
   * @description Searches for trustees meeting specified criteria (extended by iterator)
   * @param {object} params
   * @param {string} params.filters - filter string
   * @param {Array<string>} params.fields - array of strings with role attribute names to return
   * @param {Array<object>} params.sort - array of containers each of them containing two attributes
   * @param {number} params.nStart - start position for counter
   * @param {number} params.nEnd - end position for counter
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  FindTrusteesExtended ({ filters, fields, sort = [], nStart = 0, nEnd }, connection, cacheConfig) {
    var params = {
      strFilter: filters,
      pFieldsToReturn: fields,
      pFieldsToOrder: sort,
      lMaxLifeTime: 7200
    }
    return this.baseRequest({
      command: `${className}.FindTrustees`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'strAccessor'))
      .then(iteratorContainer => {
        const strAccessor = iteratorContainer.result
        const OpenApiClient = require('../open-api-client')
        return OpenApiClient.ChunkAccessor.GetItemsCount({ strAccessor }, connection)
          .then(countContainer => {
            const count = countContainer.result
            const params = {
              strAccessor: strAccessor,
              nStart: nStart,
              nCount: nEnd || count
            }
            return OpenApiClient.ChunkAccessor.GetItemsChunk(params, connection)
          })
          .then(responseContainer => {
            return OpenApiClient.ChunkAccessor.Release({ strAccessor }, connection)
              .then(() => responseContainer)
          })
      })
  }

  /**
   * @description Returns ACL for the server.
   * @param {object} params
   * @param {number} params.nId - server id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetScVServerAcl ({
    nId
  }, connection, cacheConfig) {
    let params = {
      nId
    }
    return this.baseRequest({
      command: `${className}.GetScVServerAcl`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.deserialize(body, 'pAclParams'))
  }

  /**
   * @description Set ACL for virtual server.
   * @param {object} params
   * @param {number} params.nId - virtual server id
   * @param {object} params.pAclParams - acl
   * @param {boolean} params.bCheckCurrentUserAce - if true then it checks that the user does not reduce rights for himself
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  SetScVServerAcl ({
    nId = 1,
    pAclParams,
    bCheckCurrentUserAce = false
  }, connection, cacheConfig) {
    var params = {
      nId,
      pAclParams: this.serialize(pAclParams),
      bCheckCurrentUserAce
    }
    return this.baseRequest({
      command: `${className}.SetScVServerAcl`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => {
        return this.deserialize(body, 'pAclParams')
      })
  }

  /**
   * @description Returns accessible functional areas.
   * @param {object} params
   * @param {number} params.lGroupId - id of the group
   * @param {number} params.dwAccessMask - access mask
   * @param {string} params.szwProduct - product
   * @param {string} params.szwVersion - version of product
   * @param {boolean} params.bInvert - if true then access mask is inverted
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - array of functionality areas. Each element of the array is a string: "<product>|<version>|<functional area>".
   */
  GetAccessibleFuncAreas ({
    lGroupId,
    dwAccessMask,
    szwProduct,
    szwVersion,
    bInvert
  }, connection, cacheConfig) {
    let params = {
      lGroupId,
      dwAccessMask,
      szwProduct,
      szwVersion,
      bInvert
    }
    return this.baseRequest({
      command: `${className}.GetAccessibleFuncAreas`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.deserialize(body, 'pFuncAreasArray'))
  }

  /**
   * @description Returns unaccessible functional areas.
   * @param {object} params
   * @param {number} params.lGroupId - id of the group
   * @param {string} params.szwProduct - product
   * @param {string} params.szwVersion - version of product
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - array of functionality areas. Each element of the array is a string: "<product>|<version>|<functional area>".
   */
  GetUnAccessibleFuncAreas ({
    lGroupId,
    szwProduct,
    szwVersion
  }, connection, cacheConfig) {
    let params = {
      lGroupId,
      dwAccessMask: 1,
      szwProduct,
      szwVersion,
      bInvert: true
    }
    return this.GetAccessibleFuncAreas(params, connection, cacheConfig)
      .then(res => {
        if (res.result.PxgError) {
          return res
        }
        return { result: res.result.map(item => item.split('|')[2]) }
      })
  }

  /**
   * @description Checks if current user session has access to the administration group.
   * @param {object} params
   * @param {number} params.lGroupId - id of the group
   * @param {number} params.dwAccessMask - access mask
   * @param {string} params.szwFuncArea - functional area
   * @param {string} params.szwProduct - product
   * @param {string} params.szwVersion - version of product
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} params.boolean - true if the user has access else false
   */
  AccessCheckToAdmGroup ({
    lGroupId,
    dwAccessMask,
    szwFuncArea,
    szwProduct,
    szwVersion
  }, connection, cacheConfig) {
    let params = {
      lGroupId,
      dwAccessMask,
      szwFuncArea,
      szwProduct,
      szwVersion
    }
    return this.baseRequest({
      command: `${className}.AccessCheckToAdmGroup`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.getResult(body))
  }

  /**
   * @description Returns mapping functional area to tasks.
   * @param {object} params
   * @param {string} params.szwProduct - product
   * @param {string} params.szwVersion - version of product
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - mapping of functional areas to tasks
   */
  GetMappingFuncAreaToTasks ({
    szwProduct,
    szwVersion
  }, connection, cacheConfig) {
    let params = {
      szwProduct,
      szwVersion
    }
    return this.baseRequest({
      command: `${className}.GetMappingFuncAreaToTasks`,
      data: params,
      connection
    }, cacheConfig)
      .then(body => this.getResult(body))
  }

  /**
   * Returns array of paths for all nodes actually located in the specified policy section,
   * which are readonly for current user session.
   * @param {object} params
   * @param {number} params.lGroupId - group id
   * @param {string} params.szwProduct - name of product
   * @param {string} params.szwVersion - product's version
   * @param {string} params.szwSectionName - name of setting's section name
   * @param {object} params.pPolicySection - policy's section data
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise<string[]>}
   * @constructor
   */
  GetPolicyReadonlyNodes ({
    lGroupId,
    szwProduct,
    szwVersion,
    szwSectionName,
    pPolicySection
  }, connection, cacheConfig) {
    return this
      .baseRequest({
        command: `${className}.GetPolicyReadonlyNodes`,
        data: {
          lGroupId,
          szwProduct,
          szwVersion,
          szwSectionName,
          pPolicySection
        },
        connection
      }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Returns array of paths for nodes from product's setting section,
   * which are readonly for current user session.
   * @param {object} params
   * @param {number} params.lGroupId - group id
   * @param {string} params.szwProduct - name of product
   * @param {string} params.szwVersion - product's version
   * @param {string} params.szwSectionName - name of setting's section name
   * @param {object} params.pSettingsSection - policy's section data
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise<string[]>}
   * @constructor
   */
  GetSettingsReadonlyNodes ({
    lGroupId,
    szwProduct,
    szwVersion,
    szwSectionName,
    pSettingsSection
  }, connection, cacheConfig) {
    return this
      .baseRequest({
        command: `${className}.GetSettingsReadonlyNodes`,
        data: {
          lGroupId,
          szwProduct,
          szwVersion,
          szwSectionName,
          pSettingsSection
        },
        connection
      }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Determines read only attribute by product's task type.
   * @param {object} params
   * @param {number} params.lGroupId - group id
   * @param {string} params.szwProduct - name of product
   * @param {string} params.szwVersion - product's version
   * @param {string} params.szwTaskTypeName - name of product's task type
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise<boolean>}
   * @constructor
   */
  IsTaskTypeReadonly ({
    lGroupId,
    szwProduct,
    szwVersion,
    szwTaskTypeName
  }, connection, cacheConfig) {
    return this
      .baseRequest({
        command: `${className}.IsTaskTypeReadonly`,
        data: {
          lGroupId,
          szwProduct,
          szwVersion,
          szwTaskTypeName
        },
        connection
      }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Determines read only attribute by product's task type.
   * @param {object} params
   * @param {number} params.nVServerId - server id
   * @param {number} params.llTrusteeId - Unique group/user ID
   * @param {string} params.szwProduct - name of product
   * @param {string} params.szwVersion - version of product
   * @param {object} params.pAccessRights - Access rights to check
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise<boolean>}
   * @constructor
   */

  CheckEffectiveRights ({
    nVServerId,
    llTrusteeId,
    szwProduct,
    szwVersion,
    pAccessRights
  }, connection, cacheConfig) {
    return this
      .baseRequest({
        command: `${className}.CheckEffectiveRights`,
        data: {
          nVServerId,
          llTrusteeId,
          szwProduct,
          szwVersion,
          pAccessRights: this.serialize(pAccessRights)
        },
        connection
      }, cacheConfig)
      .then(this.getResult)
  }
}
