const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'KeyService'

module.exports = class KeyService extends BaseOpenAPIClient {
  /**
   * Method unprotects crypto container created by EncryptData.
   * @param {object} params
   * @param {string} params.pEncryptedData - Encrypted data (base64 encoded)
   * @param {string} params.wstrProdName - Product name
   * @param {string} params.wstrProdVersion - Product version
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise<{string} | never>} - Decrypted data (base64 encoded)
   */
  DecryptData ({ pEncryptedData, wstrProdName, wstrProdVersion }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DecryptData`,
      data: { pEncryptedData, wstrProdName, wstrProdVersion },
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'pDecryptedData'))
  }

  /**
   * Method creates crypto container.
   * @param {object} params
   * @param {string} params.pData - Data to encrypt (Max size is 512 KB). (base64 encoded)
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise<string | never>} - Encrypted data (Crypto container format). (base64 encoded)
   */
  EncryptData ({ pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.EncryptData`,
      data: { pData },
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'pEncryptedData'))
  }

  /**
   * Method creates a crypto container for chosen host. Data may be decrypted only locally on host.
   * @param {object} params
   * @param {string} params.wstrHostId - Host id.
   * @param {string} params.pData - Data to encrypt (Max size is 512 KB). (base64 encoded)
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise<string | never>} - Encrypted data. (base64 encoded)
   */
  EncryptDataForHost ({ wstrHostId, pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.EncryptDataForHost`,
      data: { wstrHostId, pData },
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'pEncryptedData'))
  }
}
