const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'KsnInternal'

module.exports = class KsnInternal extends BaseOpenAPIClient {
  /**
   * Verify KPSN settings
   * @param {object} params
   * @param {string} params.pSettings - KPSN settings file (in PKCS7 format)
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - Params with EULA text and localization
   * @constructor
   */
  VerifyNKsnSettings ({ pSettings }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.VerifyNKsnSettings`, data: { pSettings }, connection }, cacheConfig)
      .then(this.getBody)
  }
  
  /**
   * Method verifies KPSN config and returns settings in the desired localization
   * @param {object} params
   * @param {string} params.pKpsnConfigPkcs7 - KPSN config file (in PKCS7 format)
   * @param {string} params.wstrLoc - Desired localization for eula ('en', 'ru', ...)
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - Params with EULA and others
   * @constructor
   */
  VerifyKpsnConfigLoc ({ pKpsnConfigPkcs7, wstrLoc }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.VerifyKpsnConfigLoc`, data: { pKpsnConfigPkcs7, wstrLoc }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Get KPSN eula
   * @param {object} params
   * @param {string} params.wstrNKsnLoc - Localization ('ru', 'en')
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - Params with EULA text and localization
   * @constructor
   */
  GetNKsnEula ({ wstrNKsnLoc }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.GetNKsnEula`, data: { wstrNKsnLoc }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Get all KPSN eula
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - Array of eula
   * @constructor
   */
  GetNKsnEulas (args, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.GetNKsnEulas`, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Check connection with KSN cloud (or KPSN)
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - Returns true if connection checked. Otherwise - false.
   * @constructor
   */
  CheckKsnConnection (args, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.CheckKsnConnection`, connection }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Returns settings of KsnProxy. May be used on virtual server
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - Section KSNPROXY_SETTINGS
   * @constructor
   */
  GetSettings (args, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.GetSettings`, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Send KSN statistics
   * @param {object} params
   * @param {string} params.wstrStatName - Statistics name
   * @param {object} params.pStatBody - Statistics body
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  SendKsnStatistics ({ wstrStatName, pStatBody }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SendKsnStatistics`,
      data: { wstrStatName, pStatBody: this.serialize(pStatBody) },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Check possibility to send statistics
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - returns true when possible to send statistics. Otherwise - false
   */
  NeedToSendStatistics (args, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.NeedToSendStatistics`,
      connection
    }, cacheConfig).then(this.getResult)
  }
}
