const BaseOpenAPIClient = require('../base-open-api-client')

const LRU_CACHE_MAX_SIZE = 10

module.exports = class LicenseInfoSync extends BaseOpenAPIClient {
  /**
   * The maximum number of items that remain in the cache
   * @returns {number}
   */
  get maxCacheSize () {
    return LRU_CACHE_MAX_SIZE
  }

  /**
   * Acquire serial numbers of the active and reserved licenses on the host.
   * @param {object} params
   * @param {string} params.szwHostName - Host ID
   * @param {string} params.szwProduct - Product ID
   * @param {string} params.szwVersion - Version
   * @param {Connection} connection - server connection
   * @param {CacheConfig} cacheConfig
   * @returns {Promise<*|{result: *}>}
   */
  async AcquireKeysForProductOnHost ({ szwHostName, szwProduct, szwVersion }, connection, cacheConfig) {
    return this.getBody(await this.baseRequest({
        command: `${this.constructor.name}.AcquireKeysForProductOnHost`,
        data: { szwHostName, szwProduct, szwVersion },
        connection
      }, cacheConfig))
  }

  /**
   * Acquire serial numbers of the active and reserved licenses on the host.
   * @param {object} params
   * @param {string} params.szwHostName - Host ID
   * @param {string} params.szwProduct - Product ID
   * @param {string} params.szwVersion - Version
   * @param {Connection} connection - server connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<*|{result: *}>}
   */
  async AcquireKeysForProductOnHost2 ({ szwHostName, szwProduct, szwVersion }, connection, cacheConfig) {
    return this.getResult(await this.baseRequest({
      command: `${this.constructor.name}.AcquireKeysForProductOnHost2`,
      data: { szwHostName, szwProduct, szwVersion },
      connection
    }, cacheConfig))
  }

  /**
  * Get host-specific key data.
   * @param {object} params
  * @param {string} params.szwHostName Host ID
  * @param {string} params.szwProduct Product ID
  * @param {string} params.szwVersion Version
  * @param {string} params.szwSerial Key serial number
  * @param {*} connection connection
  * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
  */
  async GetKeyDataForHost ({ szwHostName, szwProduct, szwVersion, szwSerial }, connection, cacheConfig) {
    return this.deserialize(await this.baseRequest({
      command: `${this.constructor.name}.GetKeyDataForHost`,
      data: { szwHostName, szwProduct, szwVersion, szwSerial },
      connection
    }, cacheConfig))
  }

  /**
   * Check if license is suitable for being used by the adm. server
   * @param pKeyInfo - License attribute container
   * @param bAsCurrent - true to check it as an active one, false to check it as a reserved one
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise} - Throws exception in case of error
   */
  IsLicForSaasValid2 ({ pKeyInfo, bAsCurrent }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${this.constructor.name}.IsLicForSaasValid2`,
      data: {
        pKeyInfo,
        bAsCurrent
      },
      connection
    }, cacheConfig)
  }

  /**
   * Check whether the key's product id belongs to the Public Cloud product ids list
   * @param nProductId - Product ID, mandatory
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise} - true if product id belongs to the Public Cloud product ids list, false otherwise
   */
  IsPCloudKey ({ nProductId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${this.constructor.name}.IsPCloudKey`,
      data: {
        nProductId
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Checks whether the license key activates PCloud functionality
   * @param pKeyInfo - License attribute container, mandatory
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise} - true if license key activates PCloud functionality, false otherwise
   */
  IsPCloudKey2 ({ pKeyInfo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${this.constructor.name}.IsPCloudKey2`,
      data: {
        pKeyInfo: this.serialize(pKeyInfo)
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Force synchronization of subscription licenses' metadata
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise} - Request ID used to subscribe to the event that is triggered when operation is complete
   */
  SynchronizeLicInfo2 (noParams, connection, cacheConfig) {
    return this.baseRequest({
      command: `${this.constructor.name}.SynchronizeLicInfo2`,
      data: {},
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Install adm. server's license
   * @param pKeyInfo - License attribute container
   * @param bAsCurrent - true to install it as an active one, false to install it as a reserved one
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  TryToInstallLicForSaas2 ({ pKeyInfo, bAsCurrent }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${this.constructor.name}.TryToInstallLicForSaas2`,
      data: {
        pKeyInfo,
        bAsCurrent
      },
      connection
    }, cacheConfig)
  }

  /**
   * Uninstall adm. server's license
   * @param bCurrent - true to install active license, otherwise uninstall the reserved one
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  TryToUnistallLicense ({ bCurrent }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${this.constructor.name}.TryToUnistallLicense`,
      data: {
        bCurrent
      },
      connection
    }, cacheConfig)
  }
}
