const BaseOpenApiClient = require('../base-open-api-client')
const className = 'ListTags'

module.exports = class ListTags extends BaseOpenApiClient {
  /**
   * @description Add a new tag. Allows to add a new tag value that can be set for a list item
   * @param {object} params
   * @param {string} params.tagsType - "HostsTags" || "InventoryTags" || "UmdmDeviceTags"
   * @param {string} params.szwTagValue - non-empty tag value of type
   * @param {object} params.pParams - reserved
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  AddTag ({ tagsType, szwTagValue, pParams = null } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${tagsType}.${className}.AddTag`,
      data: { szwTagValue, pParams },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }
  /**
   * @description Delete tags. Allows to delete multiple tag values in one method call
   * @param {object} params
   * @param {string} params.tagsType - "HostsTags" || "InventoryTags" || "UmdmDeviceTags"
   * @param {Array<string>} params.pTagValue - tags that will be deleted
   * @param {object} params.pParams - reserved
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  DeleteTags2 ({ tagsType, pTagValue, pParams = null } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${tagsType}.${className}.DeleteTags2`,
      data: { pTagValue, pParams },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }
  /**
   * @description Get all existing tag values. Retrieves all known tag values that can be set for a list item
   * @param {object} params
   * @param {string} params.tagsType - "HostsTags" || "InventoryTags" || "UmdmDeviceTags"
   * @param {object} params.pParams - object with attributes:
   * "KLTAGS_INCLUDE_VS" optional (boolean) flag. true(default) - to include tags information from virtual servers, false - to exclude tags information from virtual servers
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   */
  GetAllTags ({ tagsType, pParams, deserialize }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${tagsType}.${className}.GetAllTags`,
      data: { pParams },
      connection
    }, cacheConfig)
      .then(deserialize ? this.deserialize : this.getResult)
  }
  /**
   * @description Get tag values for specified list items. Retrieves tag values for specified list items
   * @param {object} params
   * @param {string} params.tagsType - "HostsTags" || "InventoryTags" || "UmdmDeviceTags"
   * @param {Array<any>} params.pIdArray - collection of list item identifiers . Value semantics depends on the list type (for HostsTags it is the host id and so on)
   * @param {object} params.pParams - reserved
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   */
  GetTags ({ tagsType, pIdArray, pParams = null }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${tagsType}.${className}.GetTags`,
      data: { pIdArray, pParams },
      connection
    }, cacheConfig).then(this.getResult)
  }
  /**
   * @description Rename a tag. Allows to rename a tag
   * @param {object} params
   * @param {string} params.tagsType - "HostsTags" || "InventoryTags" || "UmdmDeviceTags"
   * @param {string} params.szwOldTagValue - non-empty old tag value of type
   * @param {string} params.szwNewTagValue - non-empty new tag value of type
   * @param {object} params.pParams - reserved
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   */
  RenameTag ({ tagsType, szwOldTagValue, szwNewTagValue, pParams = null }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${tagsType}.${className}.RenameTag`,
      data: { szwOldTagValue, szwNewTagValue, pParams },
      connection
    }, cacheConfig).then(this.getResult)
  }
  /**
   * @description Set tag values for specified list items. Assign tag values for specified list items
   * @param {object} params
   * @param {string} params.tagsType - "HostsTags" || "InventoryTags" || "UmdmDeviceTags"
   * @param {Array<any>} params.pListItemsTags - collection of objects where each of them has the following structure:
   *  "KLTAGS_ITEM_ID" - list item identifier. Type depends on the list type
   *  "KLTAGS_TAGS" - (array) collection of (params) objects where each of them has the following structure:
   *    "KLTAGS_VALUE" - (wstring) value of a tag being set
   *    "KLTAGS_SET" - (boolean) flag. true - to set tag, false - to reset it
   * @param {object} params.pParams - object of the following structure:
   *  "KLTAGS_FULL_REPLACE" - optional (boolean) flag. true - to make a full replacement of previously set up tags, false(default) - to set up this tag and leave the old ones untouched
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  SetTags ({ tagsType, pListItemsTags, pParams = null }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${tagsType}.${className}.SetTags`,
      data: { pListItemsTags: this.serialize(pListItemsTags), pParams },
      connection
    }, cacheConfig).then(this.getResult)
  }
}
