/**
 * Migration of data between KSC On-Premise and KSCHosted.
 */
const BaseOpenApiClient = require('../base-open-api-client')
const className = 'MigrationData'

/**
 * @typedef { object } ProductEntry
 * @property { string } KLMIGR_PRODUCT_INFO_NAME - Product name
 * @property { string } KLMIGR_PRODUCT_INFO_VERSION - Product version
 * @property { string } KLMIGR_PRODUCT_INFO_DN - Product display name
 */

module.exports = class MigrationData extends BaseOpenApiClient {
  /**
   * Cancels export operation.
   * Interrupts and cancels export operation at any time
   *
   * @param { object } arg - Method arguments
   * @param { string } arg.wstrActionGuid - Async action GUID of export operation, returned by MigrationData.Export()
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  CancelExport ({ wstrActionGuid }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CancelExport`,
      data: {
        wstrActionGuid
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Performs export of objects.
   * Exports all objects, specified in pOptions, and returns async action GUID,
   * which can be used later to retrieve file URL. To provide more consistency,
   * method can also export some additional "child" objects, which are referred
   * to by "parent" objects, specified by user. As a result, user receives GUID
   * to async action, containing URL to zip-archive with all exported data
   *
   * @typedef {object} KlMigrProductInfo
   * @property {string} KLMIGR_PRODUCT_INFO_NAME - Name of product
   * @property {string} KLMIGR_PRODUCT_INFO_VERSION - Version of product
   *
   * @param { object } arg - Method arguments
   * @param { number } arg.KLMIGR_ROOT_GROUP_ID - Root group identifier
   * @param { number[] } arg.KLMIGR_ARR_ID_REPORTS - Reports identifiers
   * @param { number[] } arg.KLMIGR_ARR_ID_CMN_TASKS - Non-group tasks identifiers
   * @param { number[] } arg.KLMIGR_ARR_ID_EXTRA_QRS - Queries, which are referred from exported reports
   * @param { Array<KlMigrProductInfo> } arg.KLMIGR_ARR_PRODUCTS_INFO - array of product entries
   * @param { boolean } arg.KLMIGR_SKIP_CUSTOM_ROLES - Skip import of custom roles
   * @param { boolean } arg.KLMIGR_SKIP_USERS_GROUPS - Skip import of internal users and security groups
   * @param { boolean } arg.KLMIGR_SKIP_CUSTOM_APP_CATEGORIES - Skip import of custom application categories
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return { Promise.<object> }
   */
  Export ({
    KLMIGR_ROOT_GROUP_ID,
    KLMIGR_ARR_ID_REPORTS,
    KLMIGR_ARR_ID_CMN_TASKS,
    KLMIGR_ARR_ID_EXTRA_QRS,
    KLMIGR_ARR_PRODUCTS_INFO,
    KLMIGR_SKIP_CUSTOM_ROLES,
    KLMIGR_SKIP_USERS_GROUPS,
    KLMIGR_SKIP_CUSTOM_APP_CATEGORIES
  }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Export`,
      data: {
        pOptions: this.serialize({
          KLMIGR_ROOT_GROUP_ID,
          KLMIGR_ARR_ID_REPORTS,
          KLMIGR_ARR_ID_CMN_TASKS,
          KLMIGR_ARR_ID_EXTRA_QRS,
          KLMIGR_ARR_PRODUCTS_INFO,
          KLMIGR_SKIP_CUSTOM_ROLES,
          KLMIGR_SKIP_USERS_GROUPS,
          KLMIGR_SKIP_CUSTOM_APP_CATEGORIES
        })
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Performs import of objects.
   * Imports all objects, specified in pOptions, from file, pointed by upload
   * URL. Method is asynchronious. To correctly use this method, first call
   * InitFileUpload() to obtain file URL. If wstrUrl is invalid,
   * method fails with error
   *
   * @param { object } arg - Method arguments
   * @param { string } arg.wstrUrl - Upload URL. Use MigrationData.InitFileUpload() method to obtain it
   * @param { object } arg.pOptions - Import options
   * @param { number } arg.pOptions.KLMIGR_ROOT_GROUP_ID - Root group identifier
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return { Promise.<object> }
   */
  Import ({ wstrUrl, pOptions }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Import`,
      data: {
        wstrUrl,
        pOptions
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Acquires URL for zip archive upload.
   * Acquires URL for operation of uploading zip-archive with migration data
   * @param { null } _ - Null-argument, not used in method
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return { Promise<string> } - URL for zip-archive
   */
  InitFileUpload (_, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.InitFileUpload`,
      data: null,
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Acquire list og known products for migration
   * @param { null } _ - Null-argument, not used in method
   * @param {Connection} connection - Connection instance
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return { Promise<ProductEntry[]> }
   */
  AcquireKnownProducts (_, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AcquireKnownProducts`,
      data: null,
      connection
    }, cacheConfig).then(this.deserialize)
  }
}
