const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'PackagesApi'

module.exports = class PackagesApi extends BaseOpenAPIClient {
  /**
   * Request user agreements related to user packages, registered on current VS
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<Array<object>>} - array of UA descriptors
   * @constructor
   */
  GetUserAgreements (data, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetUserAgreements`,
      data: {},
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Requests EULA text
   * @param nEulaId - EULA id
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - EULA text
   */
  GetEulaText ({ nEulaId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEulaText`,
      data: { nEulaId },
      connection,
      raw: true
    }, cacheConfig)
      .then(this.fromBuffer)
  }

  AcceptEulas ({ vecEulaIDs = [] }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AcceptEulas`,
      data: { vecEulaIDs },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Get packages information
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<Array<object>>} - Packages array. Each item is paramParams with List of package attributes
   */
  GetPackages2 (data, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPackages2`,
      data: {},
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Get packages information
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<Array<object>>} - (array) of (paramParams) Each array entry is a paramParams container with
   * attributes described in section List of package attributes
   */
  GetPackages (data, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPackages`,
      data: {},
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Delete standalone package
   * @param nPackageId - Package ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  DeleteExecutablePkg ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteExecutablePkg`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Remove a package
   * @param nPackageId - Package ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  RemovePackage ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RemovePackage`,
      data: {
        nPackageId: +nPackageId
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Remove a package and get the list of dependent tasks
   * @param nPackageId - Package ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>} - bResult - Operation success flag
   * @returns {Promise<Array<Object>>} - pTasks - Array of dependent tasks
   */
  RemovePackage2 ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RemovePackage2`,
      data: {
        nPackageId: +nPackageId
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Get package info
   * @param nPackageId - Package ID
   * @param skipDeserialize - skip OpenAPI client deserialization
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>}
   */
  GetPackageInfo2 ({ nPackageId, skipDeserialize }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPackageInfo2`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig)
      .then(skipDeserialize ? this.getResult : this.deserialize)
  }

  /**
   * Get package info
   * @param nPackageId - package id
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} - Package info List of package attributes
   */
  GetPackageInfo ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPackageInfo`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  // eslint-disable-next-line
  SS_GetNames ({
    wstrStorageDescr,
    wstrName,
    wstrVersion,
    nTimeoutMsec = 0
  }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SS_GetNames`,
      data: {
        wstrStorageDescr,
        wstrName,
        wstrVersion,
        nTimeoutMsec
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  // eslint-disable-next-line
  SS_Read ({
    wstrStorageDescr,
    wstrName,
    wstrVersion,
    nTimeoutMsec = 0,
    wstrSection,
    skipDeserialize
  }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SS_Read`,
      data: {
        wstrStorageDescr,
        wstrName,
        wstrVersion,
        nTimeoutMsec,
        wstrSection
      },
      connection
    }, cacheConfig)
      .then(skipDeserialize ? this.getResult : this.deserialize)
  }

  // eslint-disable-next-line
  SS_Write ({
    wstrStorageDescr,
    wstrName,
    wstrVersion,
    nTimeoutMsec = 0,
    wstrSection,
    pData,
    nOperationType = 3,
    skipSerialize
  }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SS_Write`,
      data: {
        wstrStorageDescr,
        wstrName,
        wstrVersion,
        wstrSection,
        pData: skipSerialize ? pData : this.serialize(pData),
        nTimeoutMsec,
        nOperationType
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Perform operation on a settings storage section
   * @param wstrStorageDescr - Settings storage descriptor
   * @param wstrName - Settings storage name
   * @param wstrVersion - Settings storage version
   * @param nTimeoutMsec - Timeout operation
   * @param wstrSection - Settings storage section
   * @param nOperationType - Type of operation
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  // eslint-disable-next-line
  SS_SectionOperation ({
    wstrStorageDescr,
    wstrName,
    wstrVersion,
    nTimeoutMsec = 0,
    wstrSection,
    nOperationType = 3
  }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SS_SectionOperation`,
      data: {
        wstrStorageDescr,
        wstrName,
        wstrVersion,
        wstrSection,
        nTimeoutMsec,
        nOperationType
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Rename package
   * @param nPackageId - Package ID
   * @param wstrNewPackageName - New name of the package
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  RenamePackage ({
    nPackageId,
    wstrNewPackageName
  }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RenamePackage`,
      data: {
        nPackageId,
        wstrNewPackageName
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Возвращает список несовместимых с пакетом программ
   * @param nPackageId - id пакета установки
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<Array<object>} - список несовместимых программ
   * @constructor
   */
  async GetIncompatibleAppsInfo ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetIncompatibleAppsInfo`,
      data: this.serialize({
        nPackageId
      }),
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Set incompatible apps info
   * @param nPackageId - Package ID
   * @param bRemoveIncompatibleApps - Remove incompatible apps flag
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   * @constructor
   */
  async SetRemoveIncompatibleApps ({ nPackageId, bRemoveIncompatibleApps }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetRemoveIncompatibleApps`,
      data: this.serialize({
        nPackageId,
        bRemoveIncompatibleApps
      }),
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Read package file content
   * @param {object} params
   * @param {number} params.nPackageId - Package ID
   * @param {string} params.wstrFileName - File name
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<string>} pData - Binary representation of file contents
   * @constructor
   */
  async ReadPkgCfgFile ({ nPackageId, wstrFileName }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ReadPkgCfgFile`,
      data: this.serialize({
        nPackageId,
        wstrFileName
      }),
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Write package file content
   * @param {object} params
   * @param {number} params.nPackageId - Package ID
   * @param {string} params.wstrFileName - File name
   * @param {string} params.pData - Binary representation of file contents
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   * @constructor
   */
  async WritePkgCfgFile ({ nPackageId, wstrFileName, pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.WritePkgCfgFile`,
      data: this.serialize({
        nPackageId,
        wstrFileName,
        pData: pData
      }),
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Update bases in packages (asynchronously)
   * @param {object} params
   * @param {object} params.pParams - Input data container
   * @param {Array<number>} params.pParams.KLTSK_RI_PACKAGES_IDS - Packages IDs (paramArray <paramInt>, optional) If parameter
   * not exist, all packages are subject to update
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  async UpdateBasesInPackagesAsync ({ pParams }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateBasesInPackagesAsync`,
      data: {
        pParams: this.serialize(pParams)
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Cancel an asynchronous bases update operation
   * @param {object} params
   * @param {string} params.wstrRequestId - Request ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  async CancelUpdateBasesInPackages ({ wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CancelUpdateBasesInPackages`,
      data: {
        wstrRequestId
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Set license key.
   * @param {object} params
   * @param {number} params.nPackageId  Package ID.
   * @param {string} params.wstrKeyFileName Key file name.
   * @param {string} params.pData  Key data.
   * @param {boolean} params.bRemoveExisting  Remove all existing license keys.
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  async SetLicenseKey ({ nPackageId, wstrKeyFileName, pData, bRemoveExisting }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetLicenseKey`,
      data: {
        nPackageId, wstrKeyFileName, pData, bRemoveExisting
      },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Get license key.
   * @param {object} params
   * @param {number} params.nPackageId  Package ID.
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {string} params.wstrKeyFileName Key file name.
   * @returns {string} params.pMemoryChunk Key data.
   */
  async GetLicenseKey ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetLicenseKey`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Get standalone packages
   * @param nPackageId - id of installation package (returns filtered standalone packages)
   *  0 - all packages
   *  -1 - mobile packages
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<Array<object>>} - list of standalone packages
   */
  async GetExecutablePackages ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetExecutablePackages`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Publish/Unpublish a standalone package on KSC web server
   * @param bPublish - true to publish the package, false to unpublish them
   * @param nPkgExecId - Executable ID of the package
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - URL of the published package (empty string if bPublish = false).
   */
  async PublishStandalonePackage ({ bPublish, nPkgExecId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.PublishStandalonePackage`,
      data: {
        bPublish,
        nPkgExecId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Add extended certificate/sign with authenticated attributes to executable file
   * @param pParams - Extended certificate attributes, see List of standalone installation packages additional
   * signature attributes
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<datetime>} - Expiration date of certificate
   */
  async AddExtendedSign ({ pParams }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddExtendedSign`,
      data: {
        pParams
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Add extended certificate/sign with authenticated attributes to executable file (asynchronously).
   * @param pParams - Extended certificate attributes, see List of standalone installation packages additional
   * signature attributes
   * @param wstrRequestID - wstrRequestID (wstring) Request ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async AddExtendedSignAsync ({ pParams, wstrRequestID }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddExtendedSignAsync`,
      data: {
        pParams
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Allow installation of the shared prerequisites
   * @param nPackageId - Package ID
   * @param bAllow - true to allow the installation, false otherwise
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async AllowSharedPrerequisitesInstallation ({ nPackageId, bAllow }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AllowSharedPrerequisitesInstallation`,
      data: {
        nPackageId,
        bAllow
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Cancel an asynchronous call to PackagesApi.CreateExecutablePkgAsync
   * @param wstrRequestId - Request ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async CancelCreateExecutablePkg ({ wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CancelCreateExecutablePkg`,
      data: {
        wstrRequestId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Cancel an asynchronous call to PackagesApi.GetExecutablePkgFileAsync
   * @param wstrRequestId - Request ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async CancelGetExecutablePkgFile ({ wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CancelGetExecutablePkgFile`,
      data: {
        wstrRequestId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Cancel an asynchronous call to PackagesApi.RecordVapmPackageAsync or PackagesApi.RecordVapmPackageAsync
   * @param szwRequestId - Request ID of the recording process
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async CancelRecordNewPackage ({ szwRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CancelRecordNewPackage`,
      data: {
        szwRequestId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Create a standalone package (asynchronously).
   * @param {object} params
   * @param {object} params.pData
   * @param params.pData.KLTSK_RI_PACKAGES_IDS - Mandatory. Packages IDs (paramArray <paramInt>)
   * @param params.pData.KLPKG_evpExecPkgId - Standalone package ID
   * @param params.pData.KLPKG_NAME - Package name
   * @param params.pData.KLPKG_VIRTUAL - is virtual package
   * @param params.pData.KLTSK_RI_RBT_USE_TSK_SETTINGS - Ignore reboot control options from the task parameters
   * @param params.pData.KLTSK_RU_REBOOT_IMMEDIATELY - Is reboot performed immediately without user confirmation
   * @param params.pData.KLTSK_RU_REBOOT_ASK_FOR_REBOOT - Ask for reboot
   * @param params.pData.KLTSK_RU_REBOOT_ASK_FOR_REBOOT_PERIOD - Period (in minutes) of display reboot message
   * @param params.pData.KLTSK_RU_REBOOT_FORCE_REBOOT_PERIOD -  Force reboot period (in minutes)
   * @param params.pData.KLTSK_RU_REBOOT_MESSAGE -  Reboot request text
   * @param params.pData.KLTSK_RU_FORCE_APPS_CLOSED - Force applications close
   * @param params.pData.KLPKG_EXT_LOC_STRINGS - External localization strings
   * @param params.pData.UseProxy - use proxy
   * @param params.pData.ProxyServerAddress - Proxy address
   * @param params.pData.ProxyServerUser - Proxy user
   * @param params.pData.ProxyServerPassword - Proxy user password encrypted as UTF16 string
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Request ID used to subscribe to the event that is triggered when operation is complete
   */
  async CreateExecutablePkgAsync ({ pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CreateExecutablePkgAsync`,
      data: {
        pData: this.serialize(pData)
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Get standalone package file attributes (asynchronously).
   * @param pParams - Options container (reserved for future use).
   * @param nPackageId - Package ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Request ID used to subscribe to the event that is triggered when operation is complete
   */
  async GetExecutablePkgFileAsync ({ pParams, nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetExecutablePkgFileAsync`,
      data: {
        pParams,
        nPackageId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Get intranet folder for a new package
   * @param wstrProductName - Product name
   * @param wstrProductVersion - Product version
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Intranet folder path
   */
  async GetIntranetFolderForNewPackage ({ wstrProductName, wstrProductVersion }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetIntranetFolderForNewPackage`,
      data: {
        wstrProductName,
        wstrProductVersion
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Get intranet folder for particular package
   * @param nPackageId - Package ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Intranet folder path
   */
  async GetIntranetFolderForPackage ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetIntranetFolderForPackage`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Read kpd profile string
   * @param nPackageId - Package ID
   * @param wstrSection - Profile storage section
   * @param wstrKey - Profile storage key
   * @param wstrDefault - Default value
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Profile string
   */
  async GetKpdProfileString ({ nPackageId, wstrSection, wstrKey, wstrDefault }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetKpdProfileString`,
      data: {
        nPackageId,
        wstrSection,
        wstrKey,
        wstrDefault
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Get text of the login script
   * @param nPackageId - Package ID
   * @param wstrTaskId - Task ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - script text
   */
  async GetLoginScript ({ nPackageId, wstrTaskId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetLoginScript`,
      data: {
        nPackageId,
        wstrTaskId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Get information about the move-rule used by the standalone-package
   * @param nRuleId - Move-rule ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} - Move-rule information
   */
  async GetMoveRuleInfo ({ nRuleId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetMoveRuleInfo`,
      data: {
        nRuleId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Get information from archive (zip, cab, tar, tar.gz) with package data
   * @param wstrFileId - File ID of the archive file uploaded to FileTransfer
   * @param pOptions - Additional parameters
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} - Package information from archive
   */
  async GetPackageInfoFromArchive ({ wstrFileId, pOptions }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPackageInfoFromArchive`,
      data: {
        wstrFileId,
        pOptions
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Get reboot options
   * @param nPackageId - package id
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} - Reboot options
   */
  async GetRebootOptionsEx ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetRebootOptionsEx`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Check whether the package is published on KSC web server
   * @param nPkgExecId - Executable ID of the package
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>} - true if package is published, false otherwise
   */
  async IsPackagePublished ({ nPkgExecId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.IsPackagePublished`,
      data: {
        nPkgExecId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Prepare server-side data for mobile package publication on KSC web server
   * @param wstrProfileId - Profile ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - URL of the prepublished mobile package
   */
  async PrePublishMobilePackage ({ wstrProfileId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.PrePublishMobilePackage`,
      data: {
        wstrProfileId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Publish mobile manifest on KSC web server
   * @param nPkgExecId - ID of executable package
   * @param pAppData - Manifest data container
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - URL of the published manifest
   */
  async PublishMobileManifest ({ nPkgExecId, pAppData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.PublishMobileManifest`,
      data: {
        nPkgExecId,
        pAppData
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Publish mobile package on KSC web server
   * @param wstrProfileId - Profile ID
   * @param pProfileData - Package specific data container (any data)
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async PublishMobilePackage ({ wstrProfileId, pProfileData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.PublishMobileManifest`,
      data: {
        wstrProfileId,
        pProfileData
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Read kpd file
   * @param nPackageId - package id
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Read contents
   */
  async ReadKpdFile ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ReadKpdFile`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Creates a package with the default settings based on the product, overwritten in the folder, the path to which was
   * obtained by calling the
   * @param wstrPackageName - Package name
   * @param wstrFolder - Product folder (obtained by calling the PackagesApi::GetIntranetFolderForNewPackage).
   * @param wstrProductName - Product name
   * @param wstrProductVersion - Product version
   * @param wstrProductDisplName - Product display name
   * @param wstrProductDisplVersion - Product display version
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} - Container with package attributes List of package attributes.
   */
  async RecordNewPackage (
    {
      wstrPackageName,
      wstrFolder,
      wstrProductName,
      wstrProductVersion,
      wstrProductDisplName,
      wstrProductDisplVersion
    },
    connection,
    cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.RecordNewPackage`,
      data: {
        wstrPackageName, wstrFolder, wstrProductName, wstrProductVersion, wstrProductDisplName, wstrProductDisplVersion
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Creates a package with the default settings based on the product, overwritten in the folder, the path to which was
   * obtained by calling the
   * @param wstrPackageName - Package name
   * @param wstrFileId - File id
   * @param wstrProductName - Product name
   * @param wstrProductVersion - Product version
   * @param wstrProductDisplName - Product display name
   * @param wstrProductDisplVersion - Product display version
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} - Container with package attributes List of package attributes.
   */
  async RecordNewPackage2 (
    {
      wstrPackageName,
      wstrFileId,
      wstrProductName,
      wstrProductVersion,
      wstrProductDisplName,
      wstrProductDisplVersion
    },
    connection,
    cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.RecordNewPackage2`,
      data: {
        wstrPackageName, wstrFileId, wstrProductName, wstrProductVersion, wstrProductDisplName, wstrProductDisplVersion
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Create a package using data from archive (zip, cab, tar, tar.gz) or executable file. Use
   * PackagesApi::GetPackageInfoFromArchive to get package information from archive
   * @param wstrFileId - File ID of the archive or executable file uploaded to FileTransfer
   * @param pOptions - Additional parameters
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} - Package attributes List of package attributes
   */
  async RecordNewPackage3 ({ wstrFileId, pOptions }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RecordNewPackage3`,
      data: {
        wstrFileId, pOptions
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Asynchronous create a package using data from archive (zip, cab, tar, tar.gz) or executable file. Use
   * GetPackageInfoFromArchive to get package information from archive
   * @param wstrFileId - File ID of the archive or executable file uploaded to FileTransfer
   * @param pOptions - Additional parameters
   * @param wstrRequestID - Request ID used to subscribe to the event that is triggered when operation is complete
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async RecordNewPackage3Async ({ wstrFileId, pOptions, wstrRequestID }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RecordNewPackage3Async`,
      data: {
        wstrFileId, pOptions, wstrRequestID
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Creates a package (asynchronously) with the default settings based on the product, overwritten in the folder, the
   * path to which was obtained by calling the PackagesApi::GetIntranetFolderForNewPackage
   * @param wstrPackageName - Package name
   * @param wstrFolder - Product folder (obtained by calling the PackagesApi::GetIntranetFolderForNewPackage).
   * @param wstrProductName - Product name
   * @param wstrProductVersion - Product version
   * @param wstrProductDisplName - Product display name
   * @param wstrProductDisplVersion - Product display version
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Request ID used to subscribe to the event that is triggered when operation is
   * complete.
   */
  async RecordNewPackageAsync (
    {
      wstrPackageName,
      wstrFolder,
      wstrProductName,
      wstrProductVersion,
      wstrProductDisplName,
      wstrProductDisplVersion
    },
    connection,
    cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.RecordNewPackageAsync`,
      data: {
        wstrPackageName, wstrFolder, wstrProductName, wstrProductVersion, wstrProductDisplName, wstrProductDisplVersion
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Create a package using VAPM product information
   * @param {object} params
   * @param {string} params.szwNewPackageName - New package name
   * @param {object} params.parProductInfo - Product information
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Request ID used to subscribe to the event that is triggered when operation is complete
   */
  async RecordVapmPackageAsync ({ szwNewPackageName, parProductInfo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RecordVapmPackageAsync`,
      data: {
        szwNewPackageName, parProductInfo
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Reset server-specific settings for package
   * @param nPackageId - package identifier
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>} - True if reset complete successfully
   */
  async ResetDefaultServerSpecificSettings ({ nPackageId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ResetDefaultServerSpecificSettings`,
      data: {
        nPackageId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Resolve LCID of a package
   * @param nPackageId - package id
   * @param nLcid - LCID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async ResolvePackageLcid ({ nPackageId, nLcid }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ResolvePackageLcid`,
      data: {
        nPackageId,
        nLcid
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Retranslate package to a Virtual Server (asynchronously)
   * @param nPackageId - Package ID
   * @param nVServerId - Virtual server ID
   * @param pOptions - Options container
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - Request ID used to subscribe to the event that is triggered when operation is complete
   */
  async RetranslateToVServerAsync ({ nPackageId, nVServerId, pOptions }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RetranslateToVServerAsync`,
      data: {
        nPackageId, nVServerId, pOptions
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Unpublish a previously published mobile package on KSC web server
   * @param wstrProfileId - Profile ID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async UnpublishMobilePackage ({ wstrProfileId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UnpublishMobilePackage`,
      data: {
        wstrProfileId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Write kpd profile string
   * @param nPackageId - Package ID
   * @param wstrSection - Profile storage section
   * @param wstrKey - Profile storage key
   * @param wstrValue - Value to be written
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async WriteKpdProfileString ({ nPackageId, wstrSection, wstrKey, wstrValue }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.WriteKpdProfileString`,
      data: {
        nPackageId, wstrSection, wstrKey, wstrValue
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }
}
