const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'PluginDataStorage'

module.exports = class PluginDataStorage extends BaseOpenAPIClient {
  /**
   * @description Save or replace plugin data in the Plugin Data Storage.
   * @param {object} params
   * @param {string} params.wstrPluginName
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  EnumAllKeys ({wstrPluginName}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.EnumAllKeys`,
      data: {
        wstrPluginName
      },
      connection
    }, cacheConfig)
    .then(this.getResult)
  }

   /**
    * @description Return values for specified keys from the Plugin Data Storage.
    * @param {object} params
    * @param {string} params.wstrPluginName
    * @param {array} params.pArrayOfKeys
    * @param {Connection} connection
    * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
    * @returns {Promise.<object>}
    */
   LoadValues ({wstrPluginName, pArrayOfKeys}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.LoadValues`,
      data: {
        wstrPluginName,
        pArrayOfKeys
      },
      connection
    }, cacheConfig)
    .then(this.deserialize)
  }

  /**
   * @description Delete value for specified key in the Plugin Data Storage.
   * @param {object} params
   * @param {string} params.wstrPluginName
   * @param {string} params.wstrKey
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
   DeleteValue ({wstrPluginName, wstrKey}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteValue`,
      data: {
        wstrPluginName,
        wstrKey
      },
      connection
    }, cacheConfig)
    .then(this.getResult)
  }

  /**
   * @description Add or replace values for specified keys in the Plugin Data Storage.
   * @param {object} params
   * @param {string} params.wstrPluginName
   * @param {object} params.pValues
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  ReplaceValues ({wstrPluginName, pValues}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ReplaceValues`,
      data: {
        wstrPluginName,
        pValues: this.serialize(pValues)
      },
      connection
    }, cacheConfig)
    .then(this.getResult)
  }

    /**
   * @description Add or replace value URL for specified key in the Plugin Data Storage.
   * @param {object} params
   * @param {string} params.wstrPluginName
   * @param {object} params.nValueSize
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
    StartUploadValue ({ wstrPluginName, nValueSize }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.StartUploadValue`,
      data: {
        wstrPluginName,
        nValueSize
      },
      connection
    }, cacheConfig)
    .then(this.getResult)
  }

  FinishUploadValue ({ wstrUrl, wstrKey }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.FinishUploadValue`,
      data: {
        wstrUrl,
        wstrKey
      },
      connection
    }, cacheConfig)
    .then(this.getResult)
  }

  GetValue ({ url }, connection) {
    return this.sendURLRequestWithQueue({ url }, connection)
  }
}
