const BaseOpenApiClient = require('../base-open-api-client')
const className = 'PolicyProfiles'

module.exports = class PolicyProfiles extends BaseOpenApiClient {
  /**
   * Returns array of all profiles for the specified policy
   * @param {object} params
   * @param {number} params.nPolicy - policy identifier
   * @param {number} params.nRevision - policy revision id, zero means 'current policy'
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  EnumProfiles ({nPolicy, nRevision = 0} = {}, connection, cacheConfig) {
    nPolicy = parseInt(nPolicy)
    nRevision = parseInt(nRevision)
    return this.baseRequest({
      command: `${className}.EnumProfiles`,
      data: {nPolicy, nRevision},
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Returns profile data for the specified policy profile
   * @param {object} params
   * @param {number} params.nPolicy - policy id
   * @param {number} params.nRevision - policy revision id, zero means 'current policy'
   * @param {string} params.szwName - profile name, a non-empty string, up to 100 unicode characters
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  GetProfile ({nPolicy, nRevision = 0, szwName} = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetProfile`,
      data: {nPolicy, nRevision, szwName},
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Return identifier of opened SsContents, must be closed with SsContents::SS_Release
   * @param {object} params
   * @param {number} params.nPolicy - policy id
   * @param {number} params.nRevision - policy revision id, zero means 'current policy'
   * @param {string} params.szwName - profile name, a non-empty string, up to 100 unicode characters
   * @param {number} params.nLifeTime -  timeout in milliseconds to keep this SsContents object alive, zero means 'default value'
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  GetProfileSettings ({nPolicy, nRevision = 0, szwName, nLifeTime = 0} = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetProfileSettings`,
      data: {nPolicy, nRevision, szwName, nLifeTime},
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Create a new profile
   *
   * Creates a new profile with the specified name.
   * The KLSSPOL_PRF_EXPRESSION must contain a unique profile name otherwise an exception
   * KLSTD::STDE_OBJ_EXISTS is thrown.
   *
   * Actually the profile is saved only after the KLPRSSP::Contents::ApplyChanges method is called.
   * So, the KLPRSSP::Contents::ApplyChanges method may also throw an exception
   * if the profile is not unique
   * @param {object} params
   * @param {number} params.nPolicy - policy id
   * @param {string} params.szwName - profile name, a non-empty string, up to 100 unicode characters
   * @param {object} params.pAttrs - profile data, following attributes may be specified in the policy format
   * @param {number} params.nLifeTime - timeout in milliseconds to keep this SsContents object alive, zero means 'default value'
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   */
  async AddProfile ({nPolicy, szwName, pAttrs, nLifeTime = 0}, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.AddProfile`,
      data: {
        nPolicy,
        szwName,
        pAttrs: this.serialize(pAttrs),
        nLifeTime
      },
      connection
    }, cacheConfig)
    return this.getResult(result)
  }

  /**
   * Delete profile
   *
   * Deletes the specified profile and all data associated.
   * Profile with KLSSPOL_PRF_PROTECTED set to true cannot be deleted.
   * @param {object} params
   * @param {number} params.nPolicy - policy id
   * @param {string} params.szwName - profile name, a non-empty string, up to 100 unicode characters
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   */
  DeleteProfile ({nPolicy, szwName}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteProfile`,
      data: {nPolicy, szwName},
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Update attributes of an existing profile
   * Changes attributes of an existing profile.
   * @param {object} params
   * @param {number} params.nPolicy - policy id
   * @param {string} params.szwName - profile name, a non-empty string, up to 100 unicode characters
   * @param {object} params.pAttrsToUpdate - profile data, following attributes may be specified
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   */
  UpdateProfile ({nPolicy, szwName, pAttrsToUpdate}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateProfile`,
      data: {nPolicy, szwName, pAttrsToUpdate},
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Acquire profile priority array
   * Returns array of profile names, the profile with lesser index has greater priority
   * @param {object} params
   * @param {number} params.nPolicy - policy id
   * @param {number} params.nRevision - policy revision id, zero means 'current policy'
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  GetPriorities ({nPolicy, nRevision = 0}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPriorities`,
      data: {nPolicy, nRevision},
      connection
    }, cacheConfig).then(this.getResult)
  }

  PutPriorities ({nPolicy, pArrayOfNames}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.PutPriorities`,
      data: {nPolicy, pArrayOfNames},
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Переименование профиля
   * @param {object} params
   * @param {number} params.nPolicy - идентификатор политики
   * @param {string} params.szwExistingName - текущее имя пользователя
   * @param {string} params.szwNewName - новое имя пользователя
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  RenameProfile ({nPolicy, szwExistingName, szwNewName}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RenameProfile`,
      data: {nPolicy, szwExistingName, szwNewName},
      connection
    }, cacheConfig).then(this.getBody)
  }

  GetEffectivePolicyContents ({nPolicy, szwHostId, nLifeTime = 0}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEffectivePolicyContents`,
      data: {nPolicy, szwHostId, nLifeTime},
      connection
    }, cacheConfig).then(this.getResult)
  }
}
