const _ = require('lodash')
const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'Policy'

module.exports = class Policy extends BaseOpenAPIClient {
  /**
   * @description Returns policies located in specified group.
   * @param params
   * @param {number} params.nGroupId - group identifier, -1 for all
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise}
   */
  GetPoliciesForGroup ({ nGroupId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPoliciesForGroup`,
      data: { nGroupId },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Get policy general data
   * @param params
   * @param {number} params.nPolicy - policy identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  GetPolicyData ({ nPolicy }, connection, cacheConfig) {
    nPolicy = parseInt(nPolicy)
    return this.baseRequest({ command: `${className}.GetPolicyData`, data: { nPolicy }, connection }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Return identifier of opened SsContents, must be closed with SsContents::SS_Release
   * @param params
   * @param {number} params.nPolicy - policy identifier
   * @param {number} params.nRevisionId - policy revision id, 0 means 'current policy'
   * @param {number} params.nLifeTime - timeout in milliseconds to keep this SsContents object alive, zero means 'default
   *     value'
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  GetPolicyContents ({ nPolicy, nRevisionId = 0, nLifeTime = 0 } = {}, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.GetPolicyContents`, data: { nPolicy, nRevisionId, nLifeTime }, connection }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Create new policy
   * @param params
   * @param {string} params.KLPOL_DN - policy name
   * @param {string} params.KLPOL_PRODUCT - policy product alias
   * @param {string} params.KLPOL_VERSION - product version
   * @param {number} params.KLPOL_GROUP_ID - parent group id
   * @param {boolean} params.KLPOL_ACTIVE - policy activity
   * @param {boolean} params.KLPOL_ROAMING - policy out-of-office flag
   * @param {boolean} params.KLPOL_ACCEPT_PARENT - must be modified by parent policy
   * @param {boolean} params.KLPOL_FORCE_DISTRIB2CHILDREN - must modify child policies
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  AddPolicy ({
    KLPOL_DN, KLPOL_PRODUCT, KLPOL_VERSION, KLPOL_GROUP_ID, KLPOL_ACTIVE = false,
    KLPOL_ROAMING = false, KLPOL_ACCEPT_PARENT = true, KLPOL_FORCE_DISTRIB2CHILDREN = false
  } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddPolicy`,
      data: {
        pPolicyData: {
          KLPOL_DN,
          KLPOL_PRODUCT,
          KLPOL_VERSION,
          KLPOL_GROUP_ID,
          KLPOL_ACTIVE,
          KLPOL_ROAMING,
          KLPOL_ACCEPT_PARENT,
          KLPOL_FORCE_DISTRIB2CHILDREN
        }
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Update policy attributes
   */
  UpdatePolicyData ({
    nPolicy,
    pPolicyData: { KLPOL_DN, KLPOL_ACTIVE, KLPOL_ROAMING, KLPOL_ACCEPT_PARENT, KLPOL_FORCE_DISTRIB2CHILDREN }
  } = {},
  connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdatePolicyData`,
      data: {
        nPolicy,
        pPolicyData: {
          KLPOL_DN, KLPOL_ACTIVE, KLPOL_ROAMING, KLPOL_ACCEPT_PARENT, KLPOL_FORCE_DISTRIB2CHILDREN
        }
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Copies the specified policy and optionally deletes it
   * @param params
   * @param {number} params.nPolicy - policy id
   * @param {number} params.nNewGroupId - group to copy policy
   * @param {boolean} bDeleteOrigin - if server must delete nPolicy after copying
   * @param {object} pExtraData - optional container with extra attributes, may contain following data: KLPOL_DN, KLPOL_ACTIVE
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  CopyOrMovePolicy ({ nPolicy, nNewGroupId, bDeleteOrigin = false, pExtraData = null } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CopyOrMovePolicy`,
      data: {
        nPolicy, nNewGroupId, bDeleteOrigin, pExtraData
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Delete policy by id
   * @param params
   * @param {number} params.nPolicy - policy identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  DeletePolicy ({ nPolicy }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.DeletePolicy`, data: { nPolicy }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * @description Returns active and roaming policies that affect specified group
   * @param params
   * @param {number} params.nGroupId - group identifier, -1 for all
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise}
   */
  GetEffectivePoliciesForGroup ({ nGroupId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEffectivePoliciesForGroup`,
      data: { nGroupId },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * @description Revert policy to
   * Replaces the specified policy nPolicy by its revision (older version) nRevisionId
   * @param params
   * @param {number} params.nPolicyid of policy to revert
   * @param {number} params.nRevisionId id of policy revision
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise}
   */
  RevertPolicyToRevision ({ nPolicy, nRevisionId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RevertPolicyToRevision`,
      data: { nPolicy, nRevisionId },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   *
   * @description Acquire array of outbreak policies.
   * @param {*} { args } Empty param
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<Array>} Container with attribute KLPOL_POL_OUTBREAK of type paramArray, each entry contains two attributes KLPOL_ID and KLPOL_OUTBREAK_MASK
   */
  GetOutbreakPolicies ({ args }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetOutbreakPolicies`,
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   *
   * @description Specify array of outbreak policies.
   * @param params
   * @param {Array<Object>>} params.pData contains attribute KLPOL_POL_OUTBREAK of type paramArray, each entry contains two attributes KLPOL_ID and KLPOL_OUTBREAK_MASK
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  SetOutbreakPolicies ({ pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetOutbreakPolicies`,
      data: { pData },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   *
   * @description Export policy to a blob.
   * @param params
   * @param {Number} params.lPolicy policy id 
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<Blob>}
   */
  ExportPolicy ({ lPolicy }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ExportPolicy`,
      data: { lPolicy },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   *
   * @description Import policy from blob. 
   * @param params
   * @param {Number} params.pData policy exported by Policy::ExportPolicy, cannot be NULL 
   * @param {Blob} params.lGroup target group id
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} policy id
   */
  ImportPolicy ({ lGroup, pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ImportPolicy`,
      data: { lGroup, pData },
      connection
    }, cacheConfig).then(this.getResult)
  }
}
