const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'ReportManager'

module.exports = class ReportManager extends BaseOpenAPIClient {
  async EnumReports (data, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.EnumReports`,
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  async EnumReportTypes (data, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.EnumReportTypes`,
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  async GetReportTypeDetailedInfo (lReportTypeId, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetReportTypeDetailedInfo`,
      data: { lReportTypeId },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  async GetReportInfo (lReportId, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetReportInfo`,
      data: { lReportId },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  async AddReport (data, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.AddReport`,
      data: { pReportInfo: this.serialize(data) },
      connection
    }, cacheConfig)
    return this.getResult(body)
  }

  async UpdateReport ({ lReportId, data }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.UpdateReport`,
      data: {
        lReportId,
        pReportInfo: this.serialize(data)
      },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  async RemoveReport (lReportId, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.RemoveReport`,
      data: { lReportId },
      connection
    }, cacheConfig)
    return this.deserialize(body)
  }

  async ExecuteReportAsync ({ lReportId, pOptions }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.ExecuteReportAsync`,
      data: {
        lReportId,
        pOptions: this.serialize(pOptions)
      },
      connection
    }, cacheConfig)
    return this.getResult(body, 'strRequestId')
  }

  async ExecuteReportAsyncGetData (strRequestId, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.ExecuteReportAsyncGetData`,
      data: {
        strRequestId,
        nChunkSize: 16336
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  async ExecuteReportAsyncGetParsedData (strRequestId, connection, cacheConfig) {
    const data = await this.ExecuteReportAsyncGetData(strRequestId, connection, cacheConfig)

    return {
      result: {
        ...data.result,
        pXmlData: data.result.pXmlData === null
          ? null
          : Buffer.from(data.result.pXmlData, 'base64').toString('utf-8')
      }
    }
  }

  async GetDefaultReportInfo ({ lReportType }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetDefaultReportInfo`,
      data: {
        lReportType: parseInt(lReportType)
      },
      connection
    }, cacheConfig)
    return this.getResult(body, 'PxgRetVal')
  }

  async RequestStatisticsData (pRequestParams, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.RequestStatisticsData`,
      data: {
        pRequestParams: this.serialize(pRequestParams)
      },
      connection
    }, cacheConfig)
    return this.getResult(body, 'strRequestId')
  }

  async GetStatisticsData (strRequestId, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetStatisticsData`,
      data: { strRequestId },
      connection
    }, cacheConfig)
    return this.deserialize(body, 'pResultData')
  }

  async ExecuteReportAsyncCancel ({strRequestId}, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.ExecuteReportAsyncCancel`,
      data: {
        strRequestId
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  async GetConstantOutputForReportType ({ lReportType, lXmlTargetType }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetConstantOutputForReportType`,
      data: {
        lReportType,
        lXmlTargetType
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  async CreateChartPNG ({ pChartData, pOptions = { RPT_CHART_WIDTH: 300, RPT_CHART_HEIGHT: 300 } }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CreateChartPNG`,
      data: {
        pChartData: this.serialize(pChartData),
        pOptions
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Force reset of statistics data
   * Force resets statistics data, for example, resets the status of "Virus attack" or
   * "Failed to perform the administration server task" after acquaintance with the detailed information
   * @param pRequestParams - params with a statuses that need to be "reset"
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async ResetStatisticsData ({ pRequestParams }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.ResetStatisticsData`,
      data: {
        pRequestParams: this.serialize(pRequestParams)
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Enumerate ids of existing reports
   * Returns array of existing report ids
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Array<number>} array of event ids
   */
  async GetReportIds (noParams, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetReportIds`,
      data: {},
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Obtain report common data
   * Returns common data for specified report
   * @param lReportId - id of report
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {object} - report common data
   */
  async GetReportCommonData ({ lReportId }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetReportCommonData`,
      data: {
        lReportId
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Get filter settings
   * Gets filter settings
   * @param lReportType - report type
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {object} filter settings
   */
  async GetFilterSettings ({ lReportType }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetFilterSettings`,
      data: {
        lReportType
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Cancel waiting for report data from slave servers
   * @param strRequestId - identity of asynchronous operation
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async ExecuteReportAsyncCancelWaitingForSlaves ({ strRequestId }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.ExecuteReportAsyncCancelWaitingForSlaves`,
      data: {
        strRequestId
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Cancel ReportManager.RequestStatisticsData operation
   * @param strRequestId
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<*>}
   * @constructor
   */
  async CancelStatisticsRequest ({ strRequestId }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CancelStatisticsRequest`,
      data: {
        strRequestId
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Collect statistics, general statuses and dashboards.
   * Asynchronously collect statistics, general statuses and dashboards data (don't need to periodically poll,
   * data for all requested dashboards will be present in the output params).
   * Use AsyncActionStateChecker::CheckActionState to monitor state of async action.
   * @param pRequestParams
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<*>}
   * @constructor
   */
  async CollectStatisticsAsync ({ pRequestParams }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CollectStatisticsAsync`,
      data: {
        pRequestParams: this.serialize(pRequestParams)
      },
      connection
    }, cacheConfig)
    return this.getResult(body, 'wstrRequestId')
  }

  /**
   * Get result of statistics collection operation.
   * Get result of asynchronous operation ReportManager::CollectStatisticsAsync.
   * @param wstrRequestId
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<*>}
   * @constructor
   */
  async CollectStatisticsAsyncGetData ({ wstrRequestId }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.CollectStatisticsAsyncGetData`,
      data: { wstrRequestId },
      connection
    }, cacheConfig)
    return this.deserialize(body, 'pResultData')
  }
}
