const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'RetrFiles'

module.exports = class RetrFiles extends BaseOpenAPIClient {
  /**
    * Synchronously requests information about some retranslated files.
    * @param {object} params
    * @param {any[]} params.aRequest - Array of params, each cell (paramParams) contains request-info for one updatable file:
    * KLUPD_RecentIndex ("Index"): primary index relative path in lowercase, e.g. "index/u1313g.xml";
    * KLUPD_RecentCompId ("CompId"): updatable file component id in UPPERCASE, e.g. "KSC";
    * KLUPD_RecentFileName ("FileName"): file name without path in lowercase, e.g. "kscdat.zip".
    * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
    * @returns {Promise<any[] || null>} - Array of found files info, correspondingly to incoming request-array, cell-by-cell.
    * Each ouput cell is paramArray(paramParams), that is list of matched files data, where every file-info is params:
    * KLUPD_RecentRelativeSrvPath ("RelativeSrvPath"): file's relative path inside retranslation folder, e.g. "updates/ksc/".
    * If nothing is found for given in-cell, then corresponding out-cell is NULL. Null/empty in-params results in null out-array.
    */
  async GetInfo ({ aRequest }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetInfo`,
      data: {
        aRequest
      },
      connection
    }, cacheConfig)
    return this.getBody(body)
  }

  /**
   * Asynchronously requests information about some retranslated files.
   * @param aRequest - Array of params, each cell (paramParams) contains request-info in the following format:
   * KLUPD_RecentIndex ("Index"): primary index relative path in lowercase, e.g. "index/u1313g.xml";
   * KLUPD_Filter ("Filter"): updater filters in lowercase, e.g. "componentid=ksc:lang=en";
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<{result: *}>} - action guid
   * @constructor
   */
  async GetInfoAsync ({ aRequest }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetInfoAsync`,
      data: this.serialize({ aRequest }),
      connection
    }, cacheConfig)
    return this.getResult(body)
  }
}
