const _ = require('lodash')
const BaseOpenAPIClient = require('../base-open-api-client')

/**
 * Предоставляет API, состоящее из методов-хелперов, для удобной работы с Role/Roles
 */
module.exports = class Roles extends BaseOpenAPIClient {
  /**
   * Функция позволяет поплучить общую информацию о роли, а так же дополнительную:
   * - связи с пользователями и группами администрирования
   * - связи с профилями политик и группами администрирования
   * @param {object} params
   * @param {number} params.nId - role int id
   * @param {Array<string>} params.pFieldsToReturn - array of strings with role attribute names to return
   * possible field values:
   * KLHST_ACL_ROLE_ID - number - role id
   * KLHST_ACL_ROLE_NAME - string - role guid name
   * KLHST_ACL_ROLE_DN - string - role display name
   * role_products - object - role products data
   * KLHST_ACL_TRUSTEE_ID - number - trustee id
   * dn - string - Account display name
   * userPrincipalName - string - AD userPrincipalName attribute
   * objectGUID - binary - AD objectGUID attribute
   * KLHST_ACL_TRUSTEE_SID - binary - trustee SID
   * KLPOL_ID - number - policy id
   * KLPOL_DN - string - policy display name
   * KLPOL_PRODUCT - string - product name
   * KLPOL_VERSION - string - product version
   * KLPOL_GROUP_ID - number - Administration group ID where the policy is located
   * name - string - Administration group name where the policy is located
   * grp_full_name - string - Administration group full name where the policy is located
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetRoleInfo ({ nId, pFieldsToReturn } = {}, connection, cacheConfig) {
    const OpenApiClient = require('../open-api-client')
    return OpenApiClient.HstAccessControl
      .GetRole({
        nId,
        pFieldsToReturn: pFieldsToReturn || [
          'KLHST_ACL_ROLE_DN',
          'KLHST_ACL_ROLE_NAME',
          'KLHST_ACL_ROLE_ID',
          'role_products'
        ]
      }, connection, cacheConfig)
      .then(roleContainer => {
        const role = roleContainer.result
        return OpenApiClient.SrvView.Fetch({
          wstrViewName: 'PolProfileAclRolesSrvViewName',
          wstrFilter: `(KLHST_ACL_ROLE_ID = ${role.KLHST_ACL_ROLE_ID})`,
          vecFieldsToReturn: [
            'id',
            'name',
            'KLSSPOL_PRF_NAME',
            'KLPOL_ACTIVE',
            'KLPOL_PRODUCT',
            'KLPOL_ROAMING',
            'KLPOL_VERSION',
            'KLPOL_ID',
            'KLPOL_DN'
          ],
          all: 'all'
        }, connection)
          .then(profilesContainer => {
            runtime.logger.log(profilesContainer)
            const result = profilesContainer.result
            const profiles = _.get(result, 'KLCSP_ITERATOR_ARRAY', [])
            return Promise.resolve({ role, profiles })
          })
      })
      .then(info => {
        return OpenApiClient.SrvView.Fetch({
          wstrViewName: 'AdmGroupsAclRolesSrvViewName',
          wstrFilter: `(nRoleId = ${info.role.KLHST_ACL_ROLE_ID})`,
          vecFieldsToReturn: [
            'nTrusteeId',
            'name',
            'wstrTrusteeDisplayName',
            'wstrTrusteeDisplayName',
            'nGroupId',
            'wstrGroupName'
          ],
          all: 'all'
        }, connection)
          .then(usersContainer => {
            const result = usersContainer.result
            const users = _.get(result, 'KLCSP_ITERATOR_ARRAY', [])
            info['users'] = users
            runtime.logger.log(info)
            return Promise.resolve({ result: info })
          })
      })
  }
}
