const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'SecurityPolicy3'

/**
 * Allows to manage security groups of internal users.
 */
module.exports = class SecurityPolicy3 extends BaseOpenAPIClient {
  /**
   * @description Creates a security group on a server.
   * @param {object} params
   * @param {object} params.pGrpParams - parameters of a group.
   *
   * There are possible values:
   *   KLSPL_SEC_GRP_NAME (paramString) - name of a group,
   *   KLSPL_SEC_GRP_DESC (paramString) - description of a group.
   *
   * @param {number} params.lVsId - id of a virtual server, a value <= 0 means main server
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  AddSecurityGroup ({ pGrpParams, lVsId = 0 }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddSecurityGroup`,
      data: {
        pGrpParams,
        lVsId
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Adds user into a security group.
   *
   * If a group or user does not exist a error occurs.
   * Connection to a virtual server has access only to groups and users
   *   which located on this virtual server.
   *
   * A user located on a virtual server can be added only into a group
   *   located on this virtual server, otherwise a error occurs.
   *
   * A user located on a main server can be added only into a group
   *   located on this main server, otherwise a error occurs.
   * @param {object} params
   * @param {number} params.lUserId - user id
   * @param {number} params.lGrpId - security group id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  AddUserIntoSecurityGroup ({ lUserId, lGrpId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddUserIntoSecurityGroup`,
      data: {
        lUserId,
        lGrpId
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Closes user connections.
   * @param {object} params
   * @param {number} params.lUserId - user id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  CloseUserConnections ({ lUserId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CloseUserConnections`,
      data: {
        lUserId
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Delete a security group.
   *
   * Connection to a virtual server has access only to
   *   groups which located on this virtual server.
   * @param {object} params
   * @param {number} params.lGrpId - security group id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  DeleteSecurityGroup ({ lGrpId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteSecurityGroup`,
      data: { lGrpId },
      connection
    }, cacheConfig)
    .then(this.getResult)
  }

  /**
   * @description Removes user from a security group.
   *
   * Connection to a virtual server has access only to groups and users
   *   which located on this virtual server.
   * @param {object} params
   * @param {number} params.lGrpId - security group id
   * @param {number} params.lUserId - user id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  DeleteUserFromSecurityGroup ({ lGrpId, lUserId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteUserFromSecurityGroup`,
      data: {
        lGrpId,
        lUserId
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Moves user from one security group into other security group.
   * @param {object} params
   * @param {number} params.lUserId - user id
   * @param {number} params.lGrpIdFrom - security group id
   * @param {number} params.lGrpIdTo - security group id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  MoveUserIntoOtherSecurityGroup ({ lUserId, lGrpIdFrom, lGrpIdTo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.MoveUserIntoOtherSecurityGroup`,
      data: {
        lUserId,
        lGrpIdFrom,
        lGrpIdTo
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Updates a security group.
   *
   * If a group does not exist a error occurs.
   * Connection to a virtual server has access only to groups
   *   which located on this virtual server.
   * @param {object} params
   * @param {number} params.lGrpId - security group id
   * @param {object} params.pGrpParams - parameters of a group.
   *
   * There are possible values:
   *   KLSPL_SEC_GRP_NAME (paramString) - name of a group,
   *   KLSPL_SEC_GRP_DESC (paramString) - description of a group.
   *
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  UpdateSecurityGroup ({ lGrpId, pGrpParams }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateSecurityGroup`,
      data: {
        lGrpId,
        pGrpParams
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }
}
