const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'SecurityPolicy'

/**
 * Allows to manage users and permissions.
 */
module.exports = class SecurityPolicy extends BaseOpenAPIClient {
  /**
   * @description Acquire existing user properties.
   * @param {object} params
   * @param {number} params.lUserId - user id
   * @param {number} params.lVsId - user id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetUsers ({ lUserId, lVsId = 0 }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetUsers`,
      data: { lUserId, lVsId },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * @description Add new user.
   * @param {object} params
   * @param {object} params.pUser - user info, containing attributes "KLSPL_USER_*"
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  AddUser ({ pUser }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddUser`,
      data: {
        pUser
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Modify existing user properties.
   * @param {object} params
   * @param {number} params.lUserId - user id
   * @param {object} params.pUser - user info, containing attributes "KLSPL_USER_*"
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  UpdateUser ({ lUserId, pUser }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateUser`,
      data: {
        lUserId,
        pUser
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Modify internal/external user of external group properties
   * @param {object} params
   * @param {bigint} params.llTrusteeId - Unique group/user ID
   * @param {object} params.pUserData - User/group data
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  UpdateTrustee ({ llTrusteeId, pUserData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateTrustee`,
      data: {
        llTrusteeId,
        pUserData: this.serialize(pUserData)
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Delete user.
   * @param {object} params
   * @param {number} params.lUserId - user id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  DeleteUser ({ lUserId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteUser`,
      data: {
        lUserId
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Acquire current user id.
   * @param {number} lUserId - user id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetCurrentUserId (lUserId, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetCurrentUserId`,
      data: {
        lUserId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * @description Acquire current user id.
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetCurrentUserId2 (noParams, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetCurrentUserId2`,
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * @description Modify a password of the existing user..
   * @param {object} params
   * @param {number} params.lUserId - user id
   * @param {string} params.wstrOldPassword - MD5 hash of old user's password
   * @param {string} params.wstrNewPassword - MD5 hash of new user's password
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  ChangeUserPassword ({ lUserId, wstrOldPassword, wstrNewPassword }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ChangeUserPassword`,
      data: {
        lUserId,
        wstrOldPassword,
        wstrNewPassword
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * @description Save or replace current user personal data.
   * @param {object} params
   * @param {object} params.pUserData - user id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */

  SavePerUserData ({ pUserData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SavePerUserData`,
      data: {
        pUserData: this.serialize(pUserData)
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * @description Get current user personal data.
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */

  LoadPerUserData (noParams, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.LoadPerUserData`,
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * @description Save or replace personal data for proposed user.
   * @param {object} params
   * @param {number} params.lUserId user id, -1 - using binUserId
   * @param {string} params.binUserId user binary id, null - using lUserId
   * @param {number} params.nVServerId virtual server id, -1 - using connection context
   * @param {object} params.pUserData - personal user data
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */

   SavePerUserData2 ({ lUserId, binUserId, nVServerId, pUserData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SavePerUserData2`,
      data: {
        lUserId,
        binUserId,
        nVServerId,
        pUserData: this.serialize(pUserData)
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * @description Load personal data for proposed user.
   * @param {object} params
   * @param {number} params.lUserId user id, -1 - using binUserId
   * @param {string} params.binUserId user binary id, null - using lUserId
   * @param {number} params.nVServerId virtual server id, -1 - using connection context
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>} personal user data
   */

  LoadPerUserData2 ({ lUserId, binUserId, nVServerId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.LoadPerUserData2`,
      data: {
        lUserId,
        binUserId,
        nVServerId
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }
}
