const baseOpenAPIClient = require('../base-open-api-client')
const className = 'ServerHierarchy'

module.exports = class ServerHierarchy extends baseOpenAPIClient {
  /**
   * Removes slave server registration record from the master server
   * @param {object} params
   * @param {number} params.lServer - Slave server id
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  DelServer ({ lServer }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DelServer`,
      data: {
        lServer
      },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Searches for slave servers meeting specified criteria. In filter expression may be also used c_szwSrchNow.
   * @param {object} params
   * @param {string} params.wstrFilter - search filter
   * @param {Array<string>} params.pFieldsToReturn - search filter
   * @param {Array<string>} params.pFieldsToOrder - names of columns to sort by, same attributes as for pFieldsToReturn may be used
   * @param {object} params.pParams - additional params
   * @param {number} params.lMaxLifeTime - max lifetime of accessor (sec)
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   */
  FindSlaveServers ({
    wstrFilter,
    pFieldsToReturn = [],
    pFieldsToOrder = [],
    pParams = {},
    lMaxLifeTime
  }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.FindSlaveServers`,
      data: {
        wstrFilter,
        pFieldsToReturn,
        pFieldsToOrder,
        pParams,
        lMaxLifeTime
      },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Enumerates slave servers for specified administration group and returns their attributes.
   * @param {object} params
   * @param {number} params.nGroupId - group id, one may specify -1 to acquire slave servers from all groups
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<array>} - each element is object containing attributes of slave server
   */
  GetChildServers ({ nGroupId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetChildServers`,
      data: {
        nGroupId
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Returns specified attributes of given slave server.
   * @param {object} params
   * @param {number} params.lServer - Slave server id
   * @param {Array<string>} params.pFields - array of names of slave server attributes to return
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - object containing specified attributes of slave server
   */
  GetServerInfo ({ lServer, pFields }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetServerInfo`,
      data: {
        lServer,
        pFields
      },
      connection
    }, cacheConfig).then(this.deserialize)
  }

  /**
   * Register new slave server for the specified group.
   * @param {object} params
   * @param {string} params.wstrDisplName - slave server display name
   * @param {number} params.nGroupId - group id, for the slave server
   * @param {string} params.pCertificate - certificate binary data
   * @param {string} params.wstrNetAddress - server net address
   * @param {object} params.pAdditionalInfo - Additional information for the slave server
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<number>} - slave server id
   */
  RegisterServer ({ wstrDisplName, nGroupId, pCertificate, wstrNetAddress, pAdditionalInfo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RegisterServer`,
      data: {
        wstrDisplName,
        nGroupId,
        pCertificate,
        wstrNetAddress,
        pAdditionalInfo
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Modify specified attributes of given slave server
   * @param {object} params
   * @param {number} params.lServer - Slave server id
   * @param {object} params.pInfo - container with slave server attributes
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>}
   */
  UpdateServer ({ lServer, pInfo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateServer`,
      data: {
        lServer,
        pInfo
      },
      connection
    }, cacheConfig).then(this.getBody)
  }
}
