const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'ServerTransportSettings'

module.exports = class ServerTransportSettings extends BaseOpenAPIClient {
  /**
   * Sets feature active
   * @param {object} params
   * @param {string} params.szwCertType - Certificate type. For "CERT_TYPE_MOBILE" only!
   * @param {boolean} params.bFeatureActive - Should be true to activate feature
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  SetFeatureActive ({ szwCertType, bFeatureActive }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetFeatureActive`,
      data: {
        szwCertType,
        bFeatureActive
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Checks if feature is activated and certificate can be changed to some custom value
   * @param {object} params
   * @param {string} params.szwCertType - certificate type
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  IsFeatureActive ({ szwCertType }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.IsFeatureActive`,
      data: { szwCertType },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Returns current connection settings
   * @param {object} params
   * @param {string} params.szwCertType - certificate type
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  GetCurrentConnectionSettings ({ szwCertType }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetCurrentConnectionSettings`,
      data: { szwCertType },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Sets custom certificate for one of SC Server's SSL listener
   * @param {object} params
   * @param {string} params.szwCertType - certificate type
   * @param {object} params.pCertData - params with custom certificate data
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  SetCustomSrvCertificate ({ szwCertType, pCertData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetCustomSrvCertificate`,
      data: {
        szwCertType,
        pCertData
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Sets or recreates default certificate
   * @param {object} params
   * @param {string} params.szwCertType - certificate type
   * @param {object} params.pSettings
   * @param {Number} params.CERT_CHANGE_TIMEOUT - optional, seconds, certificate change timeout for reserve certificates. It should be set for reserve certificate only.
   * @param {object} params.TRSP_SETTINGS_CERT_DATA - optional. Params with certificate settings to create new certificate.
   * @param {string} params.TRSP_SETTINGS_FQDN - is in TRSP_SETTINGS_CERT_DATA, optional, certificates's FQDN, if absent, then curent (Administration Server) host's FQDN will be used instead.
   * @param {object} params.TRSP_SETTINGS_CERT - optional. Params with certificate data.
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  SetOrCreateDefaultCertificate ({ szwCertType, pSettings }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetOrCreateDefaultCertificate`,
      data: {
        szwCertType,
        pSettings: this.serialize(pSettings)
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }
}
