const baseOpenAPIClient = require('../base-open-api-client')
const className = 'SrvView'

/**
 * @typedef {object} SrvViewRecords
 * @property {Array<Object>} KLCSP_ITERATOR_ARRAY
 *
 * @typedef {object} SrvViewParams
 * @property {number} TOP_N - acquire only first N records
 * @property {boolean} USE_DISTINCT -  force using DISTINCT in select clause
 *
 */

module.exports = class SrvView extends baseOpenAPIClient {
  /**
   * Finds data records for srvview wstrViewName that satisfy conditions from filter string wstrFilter
   * and creates a server-side collection of found data records
   * @param {object} params
   * @param {string} params.wstrViewName - name of srvview
   * @param {string} params.wstrFilter - filter string, contains a condition over srvview attributes
   * @param {Array<string>} params.vecFieldsToReturn - array of srvview attribute names to return
   * @param {Array<object>} params.vecFieldsToOrder - array of containers each of them containing two attributes
   * @param {SrvViewParams} params.pParams - extra options
   * @param {number} params.lifetimeSec - max result-set lifetime in seconds
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<string>} - return wstrIteratorId
   * @constructor
   */
  ResetIterator ({ wstrViewName, wstrFilter, vecFieldsToReturn, vecFieldsToOrder = [], pParams = {}, lifetimeSec = 1000 }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ResetIterator`,
      data: {
        wstrViewName,
        wstrFilter,
        vecFieldsToReturn,
        vecFieldsToOrder,
        pParams,
        lifetimeSec
      },
      connection
    }, cacheConfig).then(body => this.getResult(body, 'wstrIteratorId'))
  }

  /**
   * Returns number of elements contained in the specified result-set
   * @param {object} params
   * @param {string} params.wstrIteratorId - result-set ID, identifier of the server-side ordered collection of found data records
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<number>}
   * @constructor
   */
  GetRecordCount ({ wstrIteratorId }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.GetRecordCount`, data: { wstrIteratorId }, connection }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Returns number of elements contained in the specified result-set
   * @param {object} params
   * @param {string} params.wstrIteratorId - result-set ID, identifier of the server-side ordered collection of found data records
   * @param {number} params.nStart - zero-based start position
   * @param {number} params.nEnd - zero-based finish position
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<SrvViewRecords>}
   * @constructor
   */
  GetRecordRange ({ wstrIteratorId, nStart, nEnd }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetRecordRange`,
      data: { wstrIteratorId, nStart, nEnd },
      connection
    }, cacheConfig)
      .then(body => this.deserialize(body, 'pRecords'))
  }

  /**
   * Releases the specified result-set and frees associated memory
   * @param {object} params
   * @param {string} params.wstrIteratorId - result-set ID, identifier of the server-side ordered collection of found data records
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<void>}
   * @constructor
   */
  ReleaseIterator ({ wstrIteratorId }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.ReleaseIterator`, data: { wstrIteratorId }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * common fetch method (wrapper function for methods of these class)
   * @param {object} params
   * @param {string} params.wstrViewName - name of srvview
   * @param {string} params.wstrFilter - filter string, contains a condition over srvview attributes
   * @param {Array<string>} params.vecFieldsToReturn - array of srvview attribute names to return
   * @param {Array<object>} params.vecFieldsToOrder - array of containers each of them containing two attributes
   * @param {SrvViewParams} params.pParams - extra options
   * @param {number} params.lifetimeSec - max result-set lifetime in seconds
   * @param {number} params.nStart - zero-based start position
   * @param {number} params.nEnd - zero-based finish position
   * @param {string} params.all - get all records
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<SrvViewRecords>}
   */
  Fetch ({
    wstrViewName,
    wstrFilter,
    vecFieldsToReturn,
    vecFieldsToOrder = [],
    pParams = null,
    lifetimeSec = 1000,
    nStart = 0,
    nEnd,
    all = true
  }, connection, cacheConfig) {
    return this.ResetIterator({
      wstrViewName,
      wstrFilter,
      vecFieldsToReturn,
      vecFieldsToOrder: this.serialize(vecFieldsToOrder),
      pParams,
      lifetimeSec
    }, connection, cacheConfig)
      .then(iteratorContainer => {
        const wstrIteratorId = iteratorContainer.result
        return this.GetRecordCount({ wstrIteratorId }, connection, cacheConfig)
          .then(countContainer => {
            const count = countContainer.result
            const params = {
              wstrIteratorId,
              nStart: all ? 0 : nStart,
              nEnd: all ? count : nEnd || count
            }
            return this.GetRecordRange(params, connection, cacheConfig)
          })
          .finally(() => {
            this.ReleaseIterator({ wstrIteratorId }, connection, cacheConfig)
          })
      })
  }
}
