const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'SsContents'

module.exports = class SsContents extends BaseOpenAPIClient {
  /**
   * Enumerate contents of settings storage
   * Retrieves list of sections.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  SS_GetNames ({ wstrID, wstrProduct, wstrVersion }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.SS_GetNames`, data: { wstrID, wstrProduct, wstrVersion }, connection }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Read data from settings storage
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {string} params.wstrSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {boolean} params.skipDeserialization
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  Ss_Read ({ wstrID, wstrProduct, wstrVersion, wstrSection, skipDeserialization }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.Ss_Read`, data: { wstrID, wstrProduct, wstrVersion, wstrSection }, connection }, cacheConfig)
      .then(skipDeserialization ? this.getBody : this.deserialize)
  }

  /**
   * Add new data to settings storage. Changes are not saved until method Ss_Apply is called.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {string} params.wstrSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {object} params.pNewData - new data to write
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  Ss_Add ({ wstrID, wstrProduct, wstrVersion, wstrSection, pNewData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Ss_Add`,
      data: { wstrID, wstrProduct, wstrVersion, wstrSection, pNewData: this.serialize(pNewData) },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Update existing data in settings storage. Changes are not saved until method Ss_Apply is called.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {string} params.wstrSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {object} params.pNewData - new data to write
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  Ss_Update ({ wstrID, wstrProduct, wstrVersion, wstrSection, pNewData }, connection, cacheConfig) {
    let data = this.serialize(pNewData)
    return this.baseRequest({
      command: `${className}.Ss_Update`,
      data: { wstrID, wstrProduct, wstrVersion, wstrSection, pNewData: data},
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Replace data in settings storage. Changes are not saved until method Ss_Apply is called.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {string} params.wstrSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {object} params.pNewData - new data to write
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  Ss_Replace ({ wstrID, wstrProduct, wstrVersion, wstrSection, pNewData, skipSerialization}, connection, cacheConfig) {
    let _pNewData = skipSerialization ? pNewData : this.serialize(pNewData)
    return this.baseRequest({
      command: `${className}.Ss_Replace`,
      data: { wstrID, wstrProduct, wstrVersion, wstrSection, pNewData: _pNewData },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Clear and write data in settings storage. Changes are not saved until method Ss_Apply is called.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {string} params.wstrSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {object} params.pNewData - new data to write
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  Ss_Clear ({ wstrID, wstrProduct, wstrVersion, wstrSection, pNewData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Ss_Clear`,
      data: { wstrID, wstrProduct, wstrVersion, wstrSection, pNewData },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Delete data from settings storage. Changes are not saved until method Ss_Apply is called.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {string} params.wstrSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {object} params.pData - data
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  Ss_Delete ({ wstrID, wstrProduct, wstrVersion, wstrSection, pData }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Ss_Delete`,
      data: { wstrID, wstrProduct, wstrVersion, wstrSection, pData },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Create section in settings storage. Changes are not saved until method Ss_Apply is called.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {string} params.wstrSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  Ss_CreateSection ({ wstrID, wstrProduct, wstrVersion, wstrSection }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Ss_CreateSection`,
      data: { wstrID, wstrProduct, wstrVersion, wstrSection },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Delete section from settings storage. Changes are not saved until method Ss_Apply is called.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {string} params.wstrProduct - product name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>
   * @param {string} params.wstrVersion - version string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {string} params.wstrSection - section name string, non-empty string, not longer than 31 character, and cannot contain characters /\:*?"<>.
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  Ss_DeleteSection ({ wstrID, wstrProduct, wstrVersion, wstrSection }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Ss_DeleteSection`,
      data: { wstrID, wstrProduct, wstrVersion, wstrSection },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Save changes.
   * @param {object} params
   * @param {string} params.wstrID - identifier of opened SsContents
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  Ss_Apply ({ wstrID }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Ss_Apply`,
      data: { wstrID },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Closes opened SsContents and releases associated server resources. After calling this method wstrID is not longer valid
   * @inheritdoc Ss_Apply
   */
  Ss_Release ({ wstrID }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.Ss_Release`, data: { wstrID }, connection }, cacheConfig)
      .then(this.getBody)
  }
}
