const _ = require('lodash')
const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'Tasks'

module.exports = class Tasks extends BaseOpenAPIClient {
  /**
   * Проверяет задачу на соответствие группе тестов.
   * Если тестовое значение - строка, то она ищется как
   * подстрока проверяемой задачи, иначе проверяется
   * полное соответствие.
   *
   * @param {object} item - Проверяемая задача
   * @param {object} tests - Набор тестов ({поле: значение})
   * @return {boolean}
   */
  testItem (item, tests) {
    for (const field in tests) {
      const searchValue = tests[field]
      const itemValue = _.get(item, field)
      if (itemValue !== undefined && typeof searchValue === 'string') {
        if (!~itemValue.toLowerCase().indexOf(searchValue.toLowerCase())) {
          return false
        }
      } else {
        if (itemValue !== searchValue) {
          return false
        }
      }
    }
    return true
  }

  ResetTasksIterator ({ nGroupId, bGroupIdSignificant = true, bIncludeSupergroups = true }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ResetTasksIterator`,
      data: {
        nGroupId,
        bGroupIdSignificant,
        bIncludeSupergroups
      },
      connection
    }, cacheConfig)
      .then(body => this.getResult(body, 'strTaskIteratorId'))
  }

  /**
   * Получение следующей задачи из выборки, основанной на итераторе
   * @param {object} props
   * @param {string} props.strTaskIteratorId - идентификатор итератора
   * @param {boolean} props.deserialize - флаг, нужна ли десериализация данных
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<TaskData>}
   */
  async getNextTask ({ strTaskIteratorId, deserialize = false }, connection, cacheConfig) {
    const result = await this.baseRequest({
      command: `${className}.GetNextTask`,
      data: { strTaskIteratorId },
      connection
    }, cacheConfig)
    const resultHandler = deserialize ? this.deserialize : this.getResult
    return resultHandler(result, 'pTaskData')
  }

  /**
   * Start specified task
   * @param {object} props
   * @param {string} props.strTask - task identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  RunTask ({ strTask }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.RunTask`, data: { strTask }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Suspend execution of the specified task
   * @param {object} params
   * @param {string} params.strTask - task identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  SuspendTask ({ strTask }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.SuspendTask`, data: { strTask }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Add target host to task
   * @param {object} params
   * @param {string} params.strTask - task identifier
   * @param {string} params.strHostName - target host identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  AddHostToTask ({ strTask, strHostName }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.AddHostToTask`, data: { strTask, strHostName }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Resume specified task
   * @param {object} params
   * @param {string} params.strTask - task identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  ResumeTask ({ strTask }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.ResumeTask`, data: { strTask }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Cancel execution of the specified task
   * @param {object} params
   * @param {string} params.strTask - task identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   * @constructor
   */
  CancelTask ({ strTask }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.CancelTask`, data: { strTask }, connection }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Delete the specified task
   * @param {object} params
   * @param {string} params.strTask - task identifier
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @constructor
   */
  DeleteTask ({ strTask }, connection, cacheConfig) {
    return this.baseRequest({ command: `${className}.DeleteTask`, data: { strTask }, connection }, cacheConfig)
      .then(this.getBody)
  }

  GetTaskData ({ taskId, skipDeserialization, strSrvObjId }, connection, cacheConfig) {
    const strTask = taskId && taskId.toString()
    return this.baseRequest({ command: `${className}.GetTaskData`, data: { strTask, strSrvObjId }, connection }, cacheConfig)
      .then(skipDeserialization ? this.getBody : this.deserialize)
  }

  AddTask ({ taskData, skipSerialization }, connection, cacheConfig) {
    const pData = skipSerialization ? taskData : this.serialize(taskData)
    return this.baseRequest({
      command: `${className}.AddTask`,
      data: { pData },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  UpdateTask ({ taskId, taskData, skipSerialization, strSrvObjId }, connection, cacheConfig) {
    const strTask = taskId && taskId.toString()
    const pData = skipSerialization ? taskData : this.serialize(taskData)
    return this.baseRequest({
      command: `${className}.UpdateTask`,
      data: {
        strTask,
        pData,
        strSrvObjId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  GetTaskStartEvent	({ strTask }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetTaskStartEvent`,
      data: { strTask },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  SetTaskStartEvent ({ taskId, startEventData, skipSerialization }, connection, cacheConfig) {
    const strTask = taskId && taskId.toString()
    const pData = skipSerialization ? startEventData : this.serialize(startEventData)
    return this.baseRequest({
      command: `${className}.SetTaskStartEvent`,
      data: {
        strTask,
        pData
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Получение конечного списка задач для группы администрирования
   * @param {object} params
   * @param {number} hostGroupId - идентификатор группы администрирования
   * @param {number} tasksCount - максимальное количество задач, которое необходимо получить. Ноль — получить все задачи
   * @param {object} searchData - данные для фильтрации списка
   * @param {boolean} deserialize - флаг, нужна ли десериализация данных
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<{result}>}
   * @constructor
   */
  async GetTasks ({ hostGroupId, tasksCount, searchData, bGroupIdSignificant, bIncludeSupergroups, deserialize = false }, connection, cacheConfig) {
    const { result: iterator } = await this.ResetTasksIterator({
      nGroupId: parseInt(hostGroupId),
      bGroupIdSignificant,
      bIncludeSupergroups
    }, connection, cacheConfig)
    const tasks = await this.DrainTasksRecursive({
      iterator,
      tasksCount,
      searchData,
      deserialize
    }, connection, undefined, cacheConfig)
    this.ReleaseTasksIterator(iterator, connection, cacheConfig)
    return {
      result: tasks
    }
  }

  ResetHostIteratorForTaskStatus ({ strTask, pFields2Return, nHostStateMask, nLifetime }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ResetHostIteratorForTaskStatus`,
      data: { strTask, pFields2Return, nHostStateMask, nLifetime },
      connection
    }, cacheConfig).then(body => this.getResult(body, 'strHostIteratorId'))
  }

  GetHostStatusRecordRange ({ strHostIteratorId, nStart, nEnd }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetHostStatusRecordRange`,
      data: { strHostIteratorId, nStart, nEnd },
      connection
    }, cacheConfig).then(body => this.getResult(body, 'pParHostStatus'))
  }

  ReleaseHostStatusIterator (strHostIteratorId, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ReleaseHostStatusIterator`,
      data: { strHostIteratorId },
      connection
    }, cacheConfig).then(this.getBody)
  }

  GetHostStatusRecordsCount (strHostIteratorId, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetHostStatusRecordsCount`,
      data: { strHostIteratorId },
      connection
    }, cacheConfig).then(this.getBody)
  }

  async GetTaskHistory (data, connection, cacheConfig) {
    if (('pSortFields' in data) && data.pSortFields.length) {
      data.pSortFields = this.serialize(data.pSortFields)
    }
    return this.baseRequest({ command: `${className}.GetTaskHistory`, data, connection }, cacheConfig)
      .then(res => this.getBody(res))
  }

  async GetTaskStatus (taskId, connection, cacheConfig) {
    const statusMapping = {
      1: 'Pending',
      2: 'Running',
      4: 'Finished Successfully',
      8: 'Finished with warning',
      16: 'Failed',
      32: 'Scheduled',
      64: 'Paused'
    }
    const { result: taskStatistics } = await this.GetTaskStatistics(taskId, connection, cacheConfig)
    if (!taskStatistics) {
      return {}
    }
    for (const bitStatus in statusMapping) {
      const stringStatus = statusMapping[bitStatus]
      taskStatistics[stringStatus] = taskStatistics[bitStatus]
      delete taskStatistics[bitStatus]
    }
    return { result: taskStatistics }
  }

  async GetTasksStatuses (tasks, connection, cacheConfig) {
    const tasksWithStatuses = []
    for (let i = 0; i < tasks.length; i++) {
      const task = tasks[i]
      const status = await this.GetTaskStatus(task.id.toString(), connection, cacheConfig)
      task.status = status.result
      tasksWithStatuses.push(task)
    }

    return { result: tasksWithStatuses }
  }

  /**
   * Рекурсивное получение задач: one by one
   * @param {object} params
   * @param {string} params.iterator - идентификатор итератора
   * @param {number} params.tasksCount - максимальное количество задач, которое необходимо получить. Ноль — получить все задачи
   * @param {object} params.searchData - данные для фильтрации списка
   * @param {boolean} params.deserialize - флаг, нужна ли десериализация данных
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @param {array} tasks
   * @return {Promise.<Array>}
   * @constructor
   */
  async DrainTasksRecursive ({ iterator, tasksCount = 0, searchData, deserialize = false }, connection, tasks = [], cacheConfig) {
    const newTask = await this.getNextTask({
      strTaskIteratorId: iterator,
      deserialize
    }, connection, cacheConfig)
    if (!newTask.result) {
      return tasks
    }
    const isInvisible = newTask.result &&
      newTask.result.TASK_ADDITIONAL_PARAMS &&
      newTask.result.TASK_ADDITIONAL_PARAMS.KLPRCI_TASK_INVISIBLE
    if ((!searchData || this.testItem(newTask.result, searchData)) && !isInvisible) {
      tasks.push(newTask.result)
    }
    let result
    if (tasksCount !== 0 && tasks.length >= tasksCount) {
      result = tasks
    } else {
      result = await this.DrainTasksRecursive({ iterator, tasksCount, searchData, deserialize }, connection, tasks, cacheConfig)
    }
    return result
  }

  GetTaskStatistics (taskId, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetTaskStatistics`,
      data: {
        strTask: taskId
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  GetNextHostStatus (iterator, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetNextHostStatus`,
      data: {
        strHostIteratorId: iterator,
        nCount: 1
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  ReleaseTasksIterator (iterator, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ReleaseTasksIterator`,
      data: {
        strTaskIteratorId: iterator
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  ProtectPassword (strPassword, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ProtectPassword`,
      data: {
        strPassword
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Update tasks credentials passwords.
   * Change password for tasks credentials.
   *
   * @param {object} props params object
   * @param {string} props.wstrUser Name of user
   * @param {string} props.wstrPass New password
   * @param {string} [props.wstrOldPass] Old password (optional)
   * @param {array} [props.pActions] tasks credentials actions (optional)
   * @param {Connection} connection current connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise} resolves with Request ID used to subscribe to the event that is triggered when operation is complete.
   */
  UpdateCredPasswords ({ wstrUser, wstrPass, wstrOldPass = '', pActions = [] }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateCredPasswords`,
      data: {
        wstrUser,
        wstrPass,
        wstrOldPass,
        pActions
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Cancel an asynchronous call to Tasks.UpdateCredPasswords.
   *
   * @param {object} param props container
   * @param {string} param.wstrRequestId - request id
   * @param {Connection} connection - current connection object
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise} operation cancelation result
   */
  CancelUpdateCredPasswords ({ wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CancelUpdateCredPasswords`,
      data: {
        wstrRequestId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Get task id by PRTS task id
   * @param {object} params
   * @param {string} params.strPrtsTaskId - PRTS task id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} task id
   */
  ResolveTaskId ({ strPrtsTaskId }, connection, cacheConfig) {
    return this.baseRequest({
        command: `${className}.ResolveTaskId`,
        data: {
          strPrtsTaskId
        },
        connection
      }, cacheConfig)
      .then(this.getResult)
  }
}
