const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'TotpGlobalSettings'

module.exports = class TotpGlobalSettings extends BaseOpenAPIClient {
  /**
   * Get global 2FA flag state
   *
   * @param {null} params - this method doesn't receive any params
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {boolean} - is 2FA enabled
   */
  async Get2FaRequiredForAll (params = null, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Get2FaRequiredForAll`,
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Set global 2FA flag state
   * @param {object} params
   * @param {boolean} params.bRequiredForAll - is 2FA enabled for all users
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async Set2FaRequiredForAll ({ bRequiredForAll }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.Set2FaRequiredForAll`,
      data: { bRequiredForAll },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Get global 2FA settings
   *
   * @param {null} params - this method doesn't receive any params
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {object} - 2FA settings
   */
  async GetTotpGlobalSettings (params = null, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetTotpGlobalSettings`,
      connection
    }, cacheConfig)

    return this.deserialize(body, 'pSettings')
  }

  /**
   * Set global 2FA settings
   * @param {object} params
   * @param {object} params.pSettings - params with new settings.
   * Supported values: c_szwTotpSettingsIssuer - TOTP Issuer name.
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async SetTotpGlobalSettings ({ pSettings } = null, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetTotpGlobalSettings`,
      data: {
        pSettings: this.serialize(pSettings)
      },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Checks whether calling user has enough access rights to modify 2FA settings
   * Used by console to decide whether to display configuration options and
   * whether to start setting up 2FA for current user.
   * 2FA settings modification is allowed only if this method returns true AND bLoggedInUsing2FA is true.
   *
   * @param {null} params - this method doesn't receive any params
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async IfCanConfigure2FaSettings (params = null, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.IfCanConfigure2FaSettings`,
      connection
    }, cacheConfig).then(this.getBody)
  }
}
