const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'TotpRegistration'

module.exports = class TotpRegistration extends BaseOpenAPIClient {
  /**
   * Get 2FA secret key
   *
   * @param {null} params - this method doesn't receive any params
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {object} secret key info
   */
  async GenerateSecret (params = null, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GenerateSecret`,
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Clears secret.
   * should be used when user canceled TOTP registration.
   * @param {object} params
   * @param {string} params.wstrSecretId - Secret ID generated by GenerateSecret Function
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async DeleteSecret ({ wstrSecretId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteSecret`,
      data: { wstrSecretId },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Set TOTP 2FA authentication for current user.
   * @param {object} params
   * @param {string} params.wstrSecretId - Secret ID generated by GenerateSecret Function
   * @param {string} params.wstrValidationCode - TOTP validation code
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async SaveSecretForCurrentUser ({ wstrSecretId, wstrValidationCode }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SaveSecretForCurrentUser`,
      data: { wstrSecretId, wstrValidationCode },
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Clears TOTP secret for current user. Function succeeds only if '2FA required for all' flag is not set,
   * or if user is in exceptions list.
   *
   * @param {null} params - this method doesn't receive any params
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async ClearSecretForCurrentUser (params, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ClearSecretForCurrentUser`,
      connection
    }, cacheConfig).then(this.getResult)
  }

  /**
   * Checks whether current user is allowed to reset his TOTP secret.
   *
   * @param {null} params - this method doesn't receive any params
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  async IfCurrentUserMayClearSecret (params, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.IfCurrentUserMayClearSecret`,
      connection
    }, cacheConfig).then(this.getResult)
  }
}
