const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'TrafficManager'

module.exports = class TrafficManager extends BaseOpenAPIClient {
  /**
   * Add traffic restriction.
   * @param {object} params
   * @param {object} params.pRestriction Restriction definition, see Traffic restrictions for details and attributes meaning
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {number} new restriction ID
   */
  async AddRestriction ({ pRestriction }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddRestriction`,
      data: { pRestriction },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Remove restriction by ID
   * @param {object} params
   * @param {Number} params.nRestrictionId Restricition ID to delete
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {*}
   */
  async DeleteRestriction ({ nRestrictionId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteRestriction`,
      data: { nRestrictionId },
      connection
    }, cacheConfig).then(this.getBody)
  }

  /**
   * Get all active restrictions list
   *
   * @param {*} args
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<Array[Object]>} {pRestrictions} array of restrictions
   */
  async GetRestrictions ({ args = null }, connection, cacheConfig) {
    const body = await this.baseRequest({
      command: `${className}.GetRestrictions`,
      connection
    }, cacheConfig)
    return this.deserialize(body, 'pRestrictions')
  }

  /**
   * Modify existing traffic restriction settings.
   * @param {object} params
   * @param {object} params.pRestriction Restriction definition, see Traffic restrictions for details and attributes meaning
   * @param {Number} params.nRestrictionId restriction to modify. If restriction with such id does not exist then new restriction will be added and this parameters will be ignored.
   * @param {*} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<Number>} modified restriction id. If restriction did not exist before call then newly created restriction id.
   */
  async UpdateRestriction ({ pRestriction, nRestrictionId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateRestriction`,
      data: { pRestriction, nRestrictionId },
      connection
    }, cacheConfig).then(this.getBody)
  }
}
