const BaseOpenApiClient = require('../base-open-api-client')
const className = 'UserDevicesApi'

module.exports = class UserDevices extends BaseOpenApiClient {
  /**
   * Add a new tag. Allows to add a new tag value that can be set for a list item
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<object>}
   */
  GetDevices ({ pUserId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetDevices`,
      data: { pUserId },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Creates the enrollment package for an user's device.
   * The method will be completed before the package is finally created, to find out
   * when the package is ready you should call method UserDevicesApi.GetEnrollmentPackage many times
   * @param {object} params
   * @param {string} params.pUserId - device owner binary identifier, empty means current user
   *        (current loginned on the server user)
   * @param {number} params.lMdmProtocols - bit mask means which protocols the enrollment package will be created
   * @param {number} params.lDeliveryType - bit mask means the enrollment package delivery type (DEPRECATED, just set 0)
   * @param {number} params.lLiveTime - means how long the enrollment package is available for download (in hours)
   * @param {string} params.c_wstrUrlFormat -  format of the unified link to download the package
   *        by a mobile device, example "https://www.ssp_server.com/pkg?id=%s"
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<bigint>}
   */
  CreateEnrollmentPackage (
      {
        pUserId,
        lMdmProtocols,
        lDeliveryType,
        lLiveTime,
        c_wstrUrlFormat
      } = {},
      connection,
      cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.CreateEnrollmentPackage`,
      data: {
        pUserId,
        lMdmProtocols,
        lDeliveryType,
        lLiveTime,
        c_wstrUrlFormat
      },
      connection
    }, cacheConfig)
      .then(this.getResult)
  }

  /**
   * Creates the enrollment package for an user's device.
   * The method will be completed before the package is finally created, to find out
   * when the package is ready you should call method UserDevicesApi.GetEnrollmentPackage many times
   * @param {object} params
   * @param {string} params.pUserId - device owner binary identifier, empty means current user
   *        (current loginned on the server user)
   * @param {number} params.lMdmProtocols - bit mask means which protocols the enrollment package will be created
   * @param {number} params.lContentType - enrollment content type:
   *        0x0001: Generate file
   *        0x0002: Generate application store URL, but in B2bCloudMode for MDM4IOS protocol
   *        will be generated a configuration profile
   * @param {number} params.lLiveTime - means how long the enrollment package is available for download (in hours)
   * @param {string} params.c_wstrUrlFormat -  format of the unified link to download the package
   *        by a mobile device, example "https://www.ssp_server.com/pkg?id=%s"
   * @param {object} params.pRecipient - recipient data such as e-mail and phone number, these data
   *        will be used to send a message with the link to the enrollment package
   * @param {object} params.pNotification - contains a welcome message and URL to download the enrollment package
   * @param {array} params.pProtSpecInfo - array of params contains protocols specific data
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<bigint>}
   */
  CreateEnrollmentPackage2 (
      {
        pUserId,
        lMdmProtocols,
        lContentType,
        lLiveTime,
        c_wstrUrlFormat,
        pRecipient,
        pNotification,
        pProtSpecInfo
      } = {},
      connection,
      cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.CreateEnrollmentPackage2`,
      data: {
        pUserId,
        lMdmProtocols,
        lContentType,
        lLiveTime,
        c_wstrUrlFormat,
        pRecipient: this.serialize(pRecipient),
        pNotification: this.serialize(pNotification),
        pProtSpecInfo
      },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Deletes a command previously posted to the specified device.
   * @param {object} params
   * @param {string} params.c_wstrCommandGuid - globally unique command instance id
   * @param {boolean} params.bForced - delete command without waiting of real removing from products
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>}
   */
  DeleteCommand ({ c_wstrCommandGuid, bForced = false } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteCommand`,
      data: { c_wstrCommandGuid, bForced },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Deletes a device from the server.
   * @param {object} params
   * @param {number} params.lDeviceId - device id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  DeleteDevice ({ lDeviceId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteDevice`,
      data: { lDeviceId },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Deletes the enrollment package.
   * @param {object} params
   * @param {bigint} params.lEnrPkgId - enrollment package id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  DeleteEnrollmentPackage ({ lEnrPkgId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteEnrollmentPackage`,
      data: { lEnrPkgId },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Generates QR code from any string
   * @param {object} params
   * @param {string} params.strInputData - input data
   * @param {number} params.lQRCodeSize - image size in pixels, for example: value 200 means image size 200x200
   * @param {number} params.lImageFormat - image format, possible value: 3 = Png
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>}
   */
  GenerateQRCode ({ strInputData, lQRCodeSize, lImageFormat } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GenerateQRCode`,
      data: { strInputData, lQRCodeSize, lImageFormat },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Acquires states of all commands posted to the specified device.
   * @param {object} params
   * @param {number} params.lDeviceId - device id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetCommands ({ lDeviceId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetCommands`,
      data: { lDeviceId },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Acquires library of all available commands.
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetCommandsLibrary ({}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetCommandsLibrary`,
      data: {},
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Makes the commands list according to bit mask of commands supported by the device
   * @param {object} params
   * @param {bigint} params.llCommandFlags - bit mask of commands supported by device
   * @param {array} params.pCommandsLibrary - array of commands from the commands library, each element
   *        is a container (paramParams) describing the command
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetDecipheredCommandList ({ llCommandFlags, pCommandsLibrary } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetDecipheredCommandList`,
      data: {
        llCommandFlags,
        pCommandsLibrary: this.serialize(pCommandsLibrary)
      },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Gets additional data from devices such as installed applications, profiles, certificates and etc.
   * @param {object} params
   * @param {array} params.pDeviceIds - array of device's integer identities
   * @param {array} params.pCategories - array of categories such as installed applications, profiles, certificates
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetDevicesExtraData ({ pDeviceIds, pCategories } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetDevicesExtraData`,
      data: {
        pDeviceIds,
        pCategories
      },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Gets information about an enrollment package created for a device.
   * @param {object} params
   * @param {bigint} params.llEnrPkgId - enrollment package id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>}
   */
  GetEnrollmentPackage ({ llEnrPkgId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEnrollmentPackage`,
      data: { llEnrPkgId },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Gets enrollment package file data.
   * @param {object} params
   * @param {string} params.c_wstrPackageId - enrollment package id extruded from HTTP request
   * @param {string} params.c_wstrPackageFileType - enrollment package file type, example: "iOS4", "iOS5",
   *        "Andr4", "WPhone2"
   * @param {number} params.lBuffOffset - start position
   * @param {number} params.lBuffSize - number of bytes to read
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>}
   */
  GetEnrollmentPackageFileData (
      {
        c_wstrPackageId,
        c_wstrPackageFileType,
        lBuffOffset,
        lBuffSize
      } = {},
      connection,
      cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.GetEnrollmentPackageFileData`,
      data: {
        c_wstrPackageId,
        c_wstrPackageFileType,
        lBuffOffset,
        lBuffSize
      },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Gets enrollment package file information if it's ready or gets a list of available file types
   * in the package if concrete file type is not recognized by the User Agent.
   * @param {object} params
   * @param {string} params.c_wstrPackageId - enrollment package id parsed from HTTP request
   * @param {string} params.c_wstrUserAgent - UserAgent string parsed from HTTP request,
   *        if c_wstrUserAgent is empty c_wstrPackageFileType will be used instead
   * @param {string} params.c_wstrPackageFileType -  enrollment package file type, example: "iOS4",
   *        "iOS5", "Andr4", "WPhone2"
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>}
   */
  GetEnrollmentPackageFileInfo (
      {
        c_wstrPackageId,
        c_wstrUserAgent,
        c_wstrPackageFileType
      } = {},
      connection,
      cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.GetEnrollmentPackageFileInfo`,
      data: {
        c_wstrPackageId,
        c_wstrUserAgent,
        c_wstrPackageFileType
      },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Gets the list of enrollment packages created for user's devices.
   * @param {object} params
   * @param {string} params.pUserId - device owner binary identifier, empty means current user
   *        (current loginned on the server user)
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetEnrollmentPackages ({ pUserId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEnrollmentPackages`,
      data: { pUserId },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Retrieves information about umdm hosts.
   * @param {object} params
   * @param {array} params.pHosts - Array of paramString of string host identifiers, no more than 100 elements.
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetHostsUmdmInfo ({ pHosts } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetHostsUmdmInfo`,
      data: this.serialize({ pHosts }),
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Returns command result data for the specific journal record.
   * @param {object} params
   * @param {bigint} params.llJrnlId - journal record id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>}
   */
  GetJournalCommandResult ({ llJrnlId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetJournalCommandResult`,
      data: { llJrnlId },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Acquires records from the journal.
   * @param {object} params
   * @param {number} params.lDeviceId - device id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetJournalRecords ({ lDeviceId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetJournalRecords`,
      data: { lDeviceId },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Acquires records from the journal without command result data.
   * @param {object} params
   * @param {number} params.lDeviceId - device id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetJournalRecords2 ({ lDeviceId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetJournalRecords2`,
      data: { lDeviceId },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Gets mobile agent setting storage data (for example SS data of the KES4Android application)
   * @param {object} params
   * @param {number} params.lDeviceId - device id
   * @param {string} params.c_wstrSectionName - required section name
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>}
   */
  GetMobileAgentSettingStorageData ({ lDeviceId, c_wstrSectionName } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetMobileAgentSettingStorageData`,
      data: { lDeviceId, c_wstrSectionName },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Retrieves multitenancy server settings.
   * @param {object} params
   * @param {string} params.c_wstrMtncServerId - string id of the multitenancy server which can be
   *        retrieved from method UserDevicesApi.GetMultitenancyServersInfo
   *        (value c_szwMtncSrvInfoStrId), if c_wstrMtncServerId is empty then settings will be
   *        retrieved from the first available multitenancy server
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>}
   */
  GetMultitenancyServerSettings ({ c_wstrMtncServerId } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetMultitenancyServerSettings`,
      data: { c_wstrMtncServerId },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Retrieves the multitenancy servers list.
   * @param {object} params
   * @param {number} params.nProtocolIds - bit mask means which multitenancy server protocols required
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetMultitenancyServersInfo ({ nProtocolIds } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetMultitenancyServersInfo`,
      data: { nProtocolIds },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Returns the array of not accepted license agreement's identifiers for mobile products.
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>}
   */
  GetNotAcceptedEulaIds ({}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetNotAcceptedEulaIds`,
      data: {},
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Retrieves group synchronization information for the UMDM policy.
   * @param {object} params
   * @param {number} params.nGroupId - Group identifier.
   * @param {bigint} params.nGSyncId - Synchronization identifier.
   * @param {array} params.pFields - array of requested attributes
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>}
   */
  GetSyncInfo ({ nGroupId, nGSyncId, pFields } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetSyncInfo`,
      data: {
        nGroupId,
        nGSyncId,
        pFields: this.serialize(pFields)
      },
      connection
    }, cacheConfig)
        .then(this.deserialize)
  }

  /**
   * Checks whether the license agreement for a mobile product is accepted.
   * @param {object} params
   * @param {string} params.binEulaUID -  EULA identity
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>}
   */
  IsEulaAccepted ({ binEulaUID } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.IsEulaAccepted`,
      data: { binEulaUID },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Posts a command to the specified device.
   * @param {object} params
   * @param {number} params.lDeviceId - device id
   * @param {string} params.c_wstrCommandGuid - globally unique command instance id.
   * @param {string} params.c_wstrCommandType - command type, you can retrieve it by calling
   *        UserDevicesApi.GetCommandsLibrary
   * @param {object} params.pArguments - command arguments
   * @param {number} params.lMdmProtocols - bit mask means which protocols will be used to process command
   *        (DEPRECATED, just set 0)
   * @param {number} params.lProcessFlags - command process flags mean algorithm to process command
   *        (DEPRECATED, just set 0)
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  PostCommand (
      {
        lDeviceId,
        c_wstrCommandGuid,
        c_wstrCommandType,
        pArguments,
        lMdmProtocols,
        lProcessFlags
      } = {},
      connection,
      cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.PostCommand`,
      data: {
        lDeviceId,
        c_wstrCommandGuid,
        c_wstrCommandType,
        pArguments: this.serialize(pArguments),
        lMdmProtocols,
        lProcessFlags
      },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Recalls a command previously posted to the specified device.
   * @param {object} params
   * @param {string} params.c_wstrCommandGuid - globally unique command instance id
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<boolean>}
   */
  RecallCommand ({ c_wstrCommandGuid } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RecallCommand`,
      data: { c_wstrCommandGuid },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Saves the license agreement for a mobile product and accepts it.
   * @param {object} params
   * @param {string} params.wstrProdName - internal name of a product for which EULA acceptance is required
   * @param {string} params.wstrProdVersion - internal version of a product for which EULA acceptance is required
   * @param {string} params.wstrProdDispName - display name of a product for which EULA acceptance is required
   * @param {string} params.wstrProdDispVersion - display version of a product for which EULA acceptance is required
   * @param {string} params.binEulaUID - EULA identity
   * @param {string} params.wstrText - EULA text
   * @param {string} params.tTimeStamp -  EULA creation time
   * @param {boolean} params.bAccept - accept EULA if value is true
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  SaveEulaAndAccept (
      {
        wstrProdName,
        wstrProdVersion,
        wstrProdDispName,
        wstrProdDispVersion,
        binEulaUID,
        wstrText,
        tTimeStamp,
        bAccept
      } = {},
      connection,
      cacheConfig
  ) {
    return this.baseRequest({
      command: `${className}.SaveEulaAndAccept`,
      data: {
        wstrProdName,
        wstrProdVersion,
        wstrProdDispName,
        wstrProdDispVersion,
        binEulaUID,
        wstrText,
        tTimeStamp,
        bAccept
      },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }

  /**
   * Sets multitenancy server settings.
   * @param {object} params
   * @param {string} params.c_wstrMtncServerId - string id of the multitenancy server which can be
   *        retrieved from method UserDevicesApi.GetMultitenancyServersInfo
   *        (value c_szwMtncSrvInfoStrId), if c_wstrMtncServerId is empty then settings will be
   *        retrieved from the first available multitenancy server
   * @param {object} params.pSettings -  contains settings to multitenancy server, now settings available
   *        only for KLUMDM::MDMProtocol_IOSMDM
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise}
   */
  SetMultitenancyServerSettings ({ c_wstrMtncServerId, pSettings } = {}, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetMultitenancyServerSettings`,
      data: {
        c_wstrMtncServerId,
        pSettings
      },
      connection
    }, cacheConfig)
        .then(this.getResult)
  }
}
