const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'VServers'

/**
 * @typedef {object} VServerType
 * @property {string} strDisplayName - virtual server display name, if display name is non-unique, it will be modified to become unique
 * @property {number} lParentGroup - virtual server parent group
 * @property {number} lVServer - virtual server id
 * @property {object} pInfo - a container containing no-read-only attributes "KLVSRV_*" to change
 * @property {number} pFields2Return - attributes "KLVSRV_*" to acquire
 * @property {object} pPermissions - ACL
 * @property {boolean} bProtection - if true checks if the user does not denies access to the server to itself
 */

module.exports = class VServers extends BaseOpenAPIClient {
  /**
   * Registers new virtual server
   * @param {VServerType} vServer
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<Object>}
   */
  AddVServerInfo ({ strDisplayName, lParentGroup }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AddVServerInfo`,
      data: {
        strDisplayName,
        lParentGroup
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Unregisters specified Virtual Server
   * @param {VServerType} vServer
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<string>} - id of asynchronous operation
   */
  DelVServer ({ lVServer }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DelVServer`,
      data: {
        lVServer
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Returns ACL for the specified virtual server
   * @param {VServerType} VServer
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<object>} - ACL
   */
  GetPermissions ({ lVServer }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPermissions`,
      data: {
        lVServer
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * @description Acquire virtual servers for the specified group. Returns array of virtual servers for the specified group.
   * @param {object} Group
   * @param {number} Group.lParentGroup - id of parent group, -1 means 'from all groups'
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<Object>}
   */
  GetVServers ({ lParentGroup = -1 }, connection, cacheConfig) {
    const params = { lParentGroup }
    return this.baseRequest({
      command: `${className}.GetVServers`,
      data: params,
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }
  /**
   * @description Acquire info on virtual server. Returns info about the specified virtual server
   * @param {VServerType} VServer
   * @param {Connection} connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @return {Promise.<Object>}
   */
  GetVServerInfo ({ lVServer, pFields2Return }, connection, cacheConfig) {
    const params = { lVServer, pFields2Return }
    return this.baseRequest({
      command: `${className}.GetVServerInfo`,
      data: params,
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Moves specified virtual server
   * @param {VServerType} vServer
   * @param {number} vServer.lNewParentGroup - new group id
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<{string}>} - id of asynchronous operation (strActionGuid)
   */
  MoveVServer ({ lVServer, lNewParentGroup }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.MoveVServer`,
      data: {
        lVServer,
        lNewParentGroup
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Function recalls Network Agent certificate from the specified virtual server and closes active connections from such Network Agents
   * @param {VServerType} vServer
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  RecallCertAndCloseConnections ({ lVServer }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.RecallCertAndCloseConnections`,
      data: {
        lVServer
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Sets ACL for the specified virtual server
   * @param {VServerType} vServer
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise.<Object>}
   */
  SetPermissions ({ lVServer, pPermissions, bProtection = true }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetPermissions`,
      data: {
        lVServer,
        pPermissions: this.serialize(pPermissions),
        bProtection
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }

  /**
   * Modifies attributes of the specified virtual server
   * @param {VServerType} vServer
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  UpdateVServerInfo ({ lVServer, pInfo }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.UpdateVServerInfo`,
      data: {
        lVServer,
        pInfo
      },
      connection
    }, cacheConfig)
      .then(this.deserialize)
  }
}
