const BaseOpenAPIClient = require('../base-open-api-client')
const className = 'VapmControlApi'

module.exports = class VapmControlApi extends BaseOpenAPIClient {
  /**
   * RAccepts given EULAs
   * @param {object} params
   * @param {Array<number>} params.pEulaIDs - Array of EULA IDs to accept
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns { Promise<object> }
   */
  AcceptEulas ({ pEulaIDs }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.AcceptEulas`,
      data: {
        pEulaIDs
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Decline given EULAs
   * @param {object} params
   * @param {Array<number>} params.pEulaIDs - Array of EULA IDs to decline
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  DeclineEulas ({ pEulaIDs }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeclineEulas`,
      data: {
        pEulaIDs
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Cancel the files cleanup process initiated by DeleteFilesForUpdates() call
   * @param wstrRequestId - request ID, used to initiate the request by the DeleteFilesForUpdates() call
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  CancelDeleteFilesForUpdates ({ wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CancelDeleteFilesForUpdates`,
      data: {
        wstrRequestId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Cancel the patch downloading started by DownloadPatchAsync()
   * @param wstrRequestId - request ID used to call DownloadPatchAsync()
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  CancelDownloadPatch ({ wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.CancelDownloadPatch`,
      data: {
        wstrRequestId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Changes updates approval
   * @param {object}
   * @param { Array<Object> } pUpdates - Array of updates to be approved/declined,
   * @param { number } nApprovementState - virtual server parent group
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns { Promise<void> }
   */
  ChangeApproval ({ pUpdates, nApprovementState }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ChangeApproval`,
      data: {
        pUpdates: this.serialize(pUpdates),
        nApprovementState
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Changes "ignore" state of a vulnerability
   * @param {object} params
   * @param { string } params.wstrVulnerabilityUid - Unique id of the vulnerability,
   * @param { string } params.wstrHostId - Host identifier for the vulnerability to be ignored on; could be empty to change the vulnerability state everywhere,
   * @param { boolean } params.bIgnore - Should the vulnerability be ignored or not
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns { Promise<void> }
   */
  ChangeVulnerabilityIgnorance ({ wstrVulnerabilityUid, wstrHostId, bIgnore }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.ChangeVulnerabilityIgnorance`,
      data: {
        wstrVulnerabilityUid,
        wstrHostId,
        bIgnore
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Cleanup all the files in all the server storages containing the bodies of the given patches. The operation progress is reported by 'KLEV_EventAsyncState' events.
   * @param pUpdatesIds - updates identities array; each entry is paramParams containing one of the following attributes: 'nPatchDbId' or 'nRevisionID'
   * @param wstrRequestId - request ID, used to cancel the request by CancelDeleteFilesForUpdates() and to subscribe for the 'KLEV_EventAsyncState' events
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  DeleteFilesForUpdates ({ pUpdatesIds, wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DeleteFilesForUpdates`,
      data: {
        pUpdatesIds, wstrRequestId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Download 3-party patch to save locally
   * @param llPatchGlbId - patch database ID
   * @param nLcid - patch Lcid
   * @param wstrRequestId - request ID, used to cancel the request by CancelDownloadPatch() or to get the result by GetDownloadPatchResult()
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   */
  DownloadPatchAsync ({ llPatchGlbId, nLcid, wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.DownloadPatchAsync`,
      data: {
        llPatchGlbId, nLcid, wstrRequestId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Returns edition of supported attributes, KLVAPM::EAttributesSetVersion
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<number>} - KLVAPM::EAttributesSetVersion value
   */
  GetAttributesSetVersionNum (noParams, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetAttributesSetVersionNum`,
      data: {},
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Get the downloaded patch body chunk
   * @param wstrRequestId - request ID used to call DownloadPatchAsync()
   * @param nStartPos - requested chunk start position
   * @param nSizeMax - maximum chunk size
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<string>} - data chunk
   */
  GetDownloadPatchDataChunk ({ wstrRequestId, nStartPos, nSizeMax }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetDownloadPatchDataChunk`,
      data: { wstrRequestId, nStartPos, nSizeMax },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Get the information on the patch download result
   * @param wstrRequestId - request ID used to call DownloadPatchAsync()
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<object>} - nSize - patch file size
   * @returns {Promise<object>} - wstrFileName - patch file name
   */
  GetDownloadPatchResult ({ wstrRequestId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetDownloadPatchResult`,
      data: { wstrRequestId },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Requests the set of EULA ids for the given set of updates
   * @param {object} params
   * @param { Array<Object> } params.pUpdates - Array of updates to request EULA's for
   * @param { number } params.nLcid - Preferred LCID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns { Promise<object> }
   */
  GetEulaParams ({ nEulaId }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEulaParams`,
      data: {
        nEulaId
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Requests the set of EULA ids for the distributives/patches which are required to install the given patch
   * @param llPatchGlobalId - VAPM patch global identity ('nPatchGlbId' update attribute)
   * @param nLCID - LCID of the patch
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>} - pEulasIds vector of EULA ids
   */
  GetEulasIdsForPatchPrerequisites ({ llPatchGlobalId, nLCID }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEulasIdsForPatchPrerequisites`,
      data: {
        llPatchGlobalId, nLCID
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Requests the set of EULA ids for the given set of updates
   * @param {object} params
   * @param { Array<Object> } params.pUpdates - Array of updates to request EULA's for
   * @param { number } params.nLcid - Preferred LCID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns { Promise<object> }
   */
  GetEulasIdsForUpdates ({ pUpdates, nLcid }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEulasIdsForUpdates`,
      data: {
        pUpdates: this.serialize(pUpdates),
        nLcid
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Requests set of EULA ids for the given set of vulnerabilities
   * @param {object} params
   * @param { Array<number> } params.pVulnerabilities - Vector of vulnerabilities ids
   * @param { number } params.nLcid - Preferred LCID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns { Promise<object> } - Vector of EULA ids
   */
  GetEulasIdsForVulnerabilitiesPatches ({ pVulnerabilities, nLcid }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEulasIdsForVulnerabilitiesPatches`,
      data: {
        pVulnerabilities,
        nLcid
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Requests the set of EULA descriptors for the given set of updates
   * @param {object} params
   * @param { Array<Object> } params.pUpdates - Array of updates to request EULA's for
   * @param { number } params.nLcid - Preferred LCID
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns { Promise<object> }
   */
  GetEulasInfo ({ pUpdates, nLcid }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetEulasInfo`,
      data: {
        pUpdates: this.serialize(pUpdates),
        nLcid
      },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Get identities of VAPM tasks which rules are still being processed
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>} - pTasksIds -  Array of task ids.
   * These tasks rules are still being processed and the information on them could not be actual within appropriate SrvView results
   * @constructor
   */
  GetPendingRulesTasks (noParams, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetPendingRulesTasks`,
      data: {},
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Gets all LCIDs supported by distributives/patches which are required to install the given patch
   * @param llPatchGlobalId - (long) VAPM patch global identity ('nPatchGlbId' update attribute)
   * @param nOriginalLcid - LCID of the original patch
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>} - pLcids - array of Lcids
   */
  GetSupportedLcidsForPatchPrerequisites ({ llPatchGlobalId, nOriginalLcid }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetSupportedLcidsForPatchPrerequisites`,
      data: { llPatchGlobalId, nOriginalLcid },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Get filter of supported languages for software update
   * @param nUpdateSource - Update source type KLVAPM::UpdateSource
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<array>} - pSupportedLanguages - Sorted array of supported languages id. Use languages
   * from pSupportedLanguages only if pSupportedLanguages not empty.
   * Otherwise use all known languages (means empty filter)
   */
  GetUpdateSupportedLanguagesFilter ({ nUpdateSource }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.GetUpdateSupportedLanguagesFilter`,
      data: { nUpdateSource },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }

  /**
   * Set custom packages as patches for a vulnerability
   * @param wstrVulnerabilityUid - Unique id of the vulnerability
   * @param pPackages - Package ids to be used as patches (in the order of presence in the order)
   * @param pParams - Additional parameters describing the custom packages assigned to fix the vulnerability;
   * currently supported: nVulnPatchPkgLCID (contains target installation LCID; 0 if suitable for any language
   * of the product to be patched).
   * @param connection
   * @param {CacheConfig} cacheConfig - config of LRU caching, can be used for enabling request method caching
   * @returns {Promise<any | never>}
   * @constructor
   */
  SetPackagesToFixVulnerability ({ wstrVulnerabilityUid, pPackages, pParams }, connection, cacheConfig) {
    return this.baseRequest({
      command: `${className}.SetPackagesToFixVulnerability`,
      data: { wstrVulnerabilityUid, pPackages, pParams },
      connection
    }, cacheConfig)
      .then(this.getBody)
  }
}
