const fs = require('fs')
const path = require('path')
const BaseOpenAPIClient = require('./base-open-api-client')
const actionsPath = path.join(__dirname, 'methods')
const mocksPath = path.join(actionsPath, 'mocks')
const utils = require('./utils')

/**
 * Взаимодействие с Open API сервером.
 * Поставляет классы для работы с OAPI.
 * Поставляется интерфейс, позволяющий передать на клиент список реализованных классов методов.
 */
class OpenAPIClient {
  constructor ({ options = {} } = {}) {
    let self = this
    this.interface = {}
    this.socketCommands = {}
    requireActionsFromDir(actionsPath)
    if (process.env.mockOpenAPI === 'true') {
      requireActionsFromDir(mocksPath)
    }

    /**
     * Require classes from dir and extend client with them.
     * @param dirPath
     */
    function requireActionsFromDir (dirPath) {
      fs.readdirSync(dirPath).forEach(function (file) {
        const filePath = path.join(dirPath, file)

        if (!fs.statSync(filePath).isFile()) return

        const OpenAPIRequestClass = require(filePath)
        const className = OpenAPIRequestClass.name

        self[className] = new OpenAPIRequestClass()
        self[className].options = options

        getOpenAPIClassMethod(OpenAPIRequestClass).forEach(methodName => {
          self.socketCommands[`${className}.${methodName}`] = OpenAPIRequestClass[methodName]
          if (self[className].isPublicInterface !== true) return
          self.interface[className] = self.interface[className] || {}
          self.interface[className][methodName] = {
            path: `${className}.${methodName}`
          }
        })
      })
    }

    // Получение списка методов класса: самого класса и его родителя, если это не базовый класс
    // Таким образо в случае наследования классов в интерфейс попадёт верный набор методов
    function getOpenAPIClassMethod (OpenAPIRequestClass) {
      let methods = Object.getOwnPropertyNames(OpenAPIRequestClass.prototype)
      const parentClass = Object.getPrototypeOf(OpenAPIRequestClass)
      if (parentClass && parentClass.prototype && parentClass.name !== BaseOpenAPIClient.name) {
        methods = methods.concat(Object.getOwnPropertyNames(parentClass.prototype))
      }
      return methods.filter(methodName => methodName !== 'constructor')
    }

    this.utils = utils
  }
}

let openAPIClient = new OpenAPIClient()

module.exports = openAPIClient
