const { prefixes } = require('@kl/constants')

/**
 * Process 401 error from KSC Proxy.
 * 1. Check status of token:
 *  1.1 If refrehsing now - retry with timeout
 *  1.2 If not refreshing go to 2
 * 2. Set refresingStatus as true
 * 3. Request new token from IAM
 * 4. Update new token in tempDataStore and connection
 * @param {any} {connection
 * @param {any} sendRequest
 * @param {any} retryCount
 * @param {any} reject}
 * @returns {Boolean} - false if processing failed, else true
 */
async function processUnauthorizedError ({ connection, sendRequest, retryCount, reject }) {
  runtime.logger.warn('[XDR] OpenApi: Got 401 status code')
  const cipher = global.sessionManager.cipher
  const isTokenRefreshingRedisKey = `${prefixes.redis.sidRefreshingTokenStatus}-${connection.clientUid}$`
  const isTokenRefreshing = await runtime.tempDataStore.get(isTokenRefreshingRedisKey)
  if (isTokenRefreshing) {
    runtime.logger.warn('[XDR] OpenApi: Got 401 status code, token is already refreshing. Retry')
    const retryTimeout = process.env.XDR_OPEN_API_RETRY_TIMEOUT || 1000
    const maxRetryCount = process.env.XDR_OPEN_API_RETRY_COUNT || 5
    retryCount++
    if (retryCount <= maxRetryCount) {
      runtime.logger.warn(`[XDR] OpenApi: Got 401 status code. Retry #${retryCount} / ${maxRetryCount}`)
      setTimeout(async () => {
        const oAuthTokenEncrypted = await runtime.tempDataStore.get(`${prefixes.redis.sidToOAuthToken}-${connection.clientUid}$`)
        if (oAuthTokenEncrypted) {
          const oAuthTokenDecrypted = cipher.decrypt(oAuthTokenEncrypted)
          const oAuthToken = JSON.parse(oAuthTokenDecrypted)
          runtime.logger.log('[XDR] OpenApi: get oAuthToken from tempDataStore, update oAuthToken in connection.')
          connection.updateToken(oAuthToken)
        } else {
          runtime.logger.log('[XDR] OpenApi: no oAuthToken found, unable to send request. Try again.')
        }
        sendRequest()
      }, retryTimeout)
    } else {
      reject(new Error('[XDR] OpenApi: all attempts to send request have been used. Unable to send request.'))
      return false
    }
  } else {
    try {
      runtime.logger.warn('[XDR] OpenApi: Got 401 status code, request new token')
      const MINUTE = 60
      const isTokenRefreshingStatusLifetime = 5 * MINUTE
      await runtime.tempDataStore.set(
        isTokenRefreshingRedisKey,
        true,
        { timeout: isTokenRefreshingStatusLifetime }
      )
      const tokenService = await runtime.get('XDR.tokenService')
      const refreshToken = connection.oAuthToken.refresh_token
      const newToken = await tokenService.refreshToken({ refreshToken })
      if (newToken instanceof Error) {
        runtime.logger.error('[XDR] OpenApi: an error occured while refresh token. Close connection.')
        connection.callEventHandlers('error')
        connection.callEventHandlers('close')
        const err = newToken
        reject(err)
        return false
      }
      connection.updateToken(newToken)
      const TOKEN_LIFETIME_X2 = 7200
      await runtime.tempDataStore.set(
        `${prefixes.redis.sidToOAuthToken}-${connection.clientUid}$`,
        cipher.encrypt(JSON.stringify(newToken)),
        { timeout: (Number(process.env.XDR_IAM_TOKEN_LIFETIME) * 2) || TOKEN_LIFETIME_X2 }
      )
      await runtime.tempDataStore.delete(isTokenRefreshingRedisKey)
      sendRequest()
      return false
    } catch (e) {
      runtime.logger.error('[XDR] OpenApi: refresh token failed:', e)
    }
  }
  return true
}

module.exports = processUnauthorizedError
